﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeSvcVeneer
{
    internal class Program
    {
        public static CommandLineOption Options { get; private set; }

        private static CodeGenParams SelectUserAbiArchitecture(CommandLineOption.AbiTypes a)
        {
            switch (a)
            {
            case CommandLineOption.AbiTypes.Arm333: return CodeGenParams.AArch32Abi;
            case CommandLineOption.AbiTypes.Arm336: return CodeGenParams.AArch32Abi;
            case CommandLineOption.AbiTypes.Arm366: return CodeGenParams.AArch64P32Abi;
            case CommandLineOption.AbiTypes.Arm666: return CodeGenParams.AArch64P64Abi;
            }
            throw new ErrorException("内部エラー");
        }
        private static CodeGenParams SelectKernelAbiArchitecture(CommandLineOption.AbiTypes a)
        {
            switch (a)
            {
            case CommandLineOption.AbiTypes.Arm333: return CodeGenParams.AArch32Abi;
            case CommandLineOption.AbiTypes.Arm336: return CodeGenParams.AArch64P64Abi;
            case CommandLineOption.AbiTypes.Arm366: return CodeGenParams.AArch64P64Abi;
            case CommandLineOption.AbiTypes.Arm666: return CodeGenParams.AArch64P64Abi;
            }
            throw new ErrorException("内部エラー");
        }
        private static CodeGenParams SelectSvcArchitecture(CommandLineOption.AbiTypes a)
        {
            switch (a)
            {
            case CommandLineOption.AbiTypes.Arm333: return CodeGenParams.AArch32Svc;
            case CommandLineOption.AbiTypes.Arm336: return CodeGenParams.AArch32Svc;
            case CommandLineOption.AbiTypes.Arm366: return CodeGenParams.AArch64Svc;
            case CommandLineOption.AbiTypes.Arm666: return CodeGenParams.AArch64Svc;
            }
            throw new ErrorException("内部エラー");
        }
        private static CodeGenNames SelectNames(CommandLineOption.AbiTypes a)
        {
            switch (a)
            {
            case CommandLineOption.AbiTypes.Arm333: return CodeGenNames.Arm333;
            case CommandLineOption.AbiTypes.Arm336: return CodeGenNames.Arm336;
            case CommandLineOption.AbiTypes.Arm366: return CodeGenNames.Arm366;
            case CommandLineOption.AbiTypes.Arm666: return CodeGenNames.Arm666;
            }
            throw new ErrorException("内部エラー");
        }
        private static CodeGenNames SelectNamesProfiler(CommandLineOption.AbiTypes a)
        {
            switch (a)
            {
                case CommandLineOption.AbiTypes.Arm333: return CodeGenNames.Arm333p;
                case CommandLineOption.AbiTypes.Arm336: return CodeGenNames.Arm336p;
                case CommandLineOption.AbiTypes.Arm366: return CodeGenNames.Arm366p;
                case CommandLineOption.AbiTypes.Arm666: return CodeGenNames.Arm666p;
            }
            throw new ErrorException("内部エラー");
        }

        public static void Main(string[] args)
        {
            Console.OutputEncoding = Encoding.UTF8;
            AppDomain.CurrentDomain.UnhandledException += CurrentDomain_UnhandledException;

            try
            {
                Options = new CommandLineOption(args);

                if (Options.DefinitionPath == null)
                {
                    throw new ErrorException("SVC 定義ファイルが指定されていません。");
                }
                if (Options.TemplatePath == null)
                {
                    throw new ErrorException("テンプレートファイルが指定されていません。");
                }
                if (Options.OutRootPath == null)
                {
                    throw new ErrorException("出力パスが指定されていません。");
                }
                if (Options.Abi == CommandLineOption.AbiTypes.NotSpecified)
                {
                    throw new ErrorException("ABI が指定されていません。");
                }

                var aAbiUser    = SelectUserAbiArchitecture     (Options.Abi);
                var aAbiKernel  = SelectKernelAbiArchitecture   (Options.Abi);
                var aSvc        = SelectSvcArchitecture         (Options.Abi);
                var names       = SelectNames                   (Options.Abi);

                SvcSet set;

                try
                {
                    // 定義ファイルを読み込み
                    var source = File.ReadAllText(Options.DefinitionPath);

                    // パースして構文木を取得
                    var parsedSource = Parser.SvcParserHelper.Parse(source, Options.DefinitionPath);

                    // 構文木から扱いやすいデータ構造に整理
                    set = SvcSet.Construct(
                        parsedSource,
                        aSvc.PointerSize,
                        aSvc.RegisterSize,
                        string.Format("{0}::{1}", CodeGenNames.UserCommonNamespace, names.TypeNamespace),
                        CodeGenNames.UserCommonNamespace);
                }
                catch (DirectoryNotFoundException e)
                {
                    throw new ErrorException("ディレクトリが見つかりませんでした。", e);
                }
                catch (FileNotFoundException e)
                {
                    throw new ErrorException("ファイルが見つかりませんでした。", e);
                }

                // ABI レイアウトを作成
                var abiUser = AbiLayout.Arrange(set.Operations, aAbiUser);
                var abiKernel = AbiLayout.Arrange(set.Operations, aAbiKernel);

                // システムコールレイアウトを作成
                var svc = SvcLayout.Arrange(set.Operations, abiUser, aSvc);

                // レイアウト変換処理を生成
                var conversionUser = LayoutConversion.Generate(abiUser, svc);
                var conversionKernel = LayoutConversion.Generate(abiKernel, svc);

                var pathSet = new PathSet(Options);

                // コード生成
                {
                    var pNames = SelectNamesProfiler(Options.Abi);
                    var g = new SvcStubSourceGenerator(pNames);
                    g.Generate(set, abiUser, svc, conversionUser, Options.TemplatePath, pathSet.ProfilerStubSourePathTemplate, pathSet.IdIncludeTemplate);
                }
                {
                    var g = new SvcProfilerSourceGenerator(names);
                    g.Generate(set, abiUser, conversionUser, Options.TemplatePath, pathSet.ProfilerSourcePath);
                }
            }
            catch (ErrorException ee)
            {
                if (ee.InnerException != null)
                {
                    if (ee.InnerException is ErrorException)
                    {
                        PrintError((ErrorException)ee.InnerException);
                        Util.Log(string.Empty);
                    }
                    else
                    {
                        Util.Log("{0}\n", ee.InnerException.Message);
                    }
                }
                PrintError(ee);

                Environment.Exit(1);
            }
        }

        private static void PrintError(ErrorException ee)
        {
            Util.Log(ee.Message);
            if (ee.DetailMessage != null)
            {
                Util.Log(string.Empty);
                Util.Log(ee.DetailMessage);
            }
        }

        private static void CurrentDomain_UnhandledException(object sender, UnhandledExceptionEventArgs e)
        {
            var sb = new StringBuilder();
            Util.Log(string.Empty);
            Util.Log("----------------------------");
            Util.Log("内部エラー");
            Util.Log("  {0}", e.ExceptionObject.GetType().Name);
            if (e.ExceptionObject is Exception)
            {
                var exp = e.ExceptionObject as Exception;
                var st = exp.StackTrace;
                var end = st.IndexOfLineEnd(0);

                Util.Log("  {0}", exp.Message);
                Util.Log("  {0}", st);
            }
            Util.Log("----------------------------");

            {
                Debugger.Launch();
            }
        }
    }
}
