﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeSvcVeneer
{
    internal class AArch32CodeGenerator : CodeGenerator
    {
        private static string[] s_registerNameMap =
        {
            "r0", "r1", "r2", "r3",
            "r4", "r5", "r6", "r7",
            "r8", "r9", "r10", "r11",
            "r12", "sp", "lr", "pc"
        };

        public override void CallSystem(string id)
        {
            this.AddCode("svc", string.Format("#{0}", id));
        }

        public override void CallFunction(string name)
        {
            this.AddCode("bl", name);
        }

        public override void Return()
        {
            this.AddCode("bx", "lr");
        }

        public override void AllocateFromStack(int size)
        {
            this.AddCode("sub", string.Format("sp, sp, #{0}", size));
        }

        public override void FreeToStack(int size)
        {
            this.AddCode("add", string.Format("sp, sp, #{0}", size));
        }

        public override void SaveRegisters(int[] regs, bool withLr, bool isRequirePadding)
        {
            IEnumerable<int> registers = regs;
            if (isRequirePadding)
            {
                registers = registers.Add(12);
            }
            if (withLr)
            {
                registers = registers.Add(14);
            }
            var regList = string.Join(",", registers.Select(x => s_registerNameMap[x]));
            this.AddCode("push", "{" + regList + "}");
        }

        public override void RestoreRegisters(int[] regs, bool withPc, bool isRequirePadding)
        {
            IEnumerable<int> registers = regs;
            if (isRequirePadding)
            {
                registers = registers.Add(12);
            }
            if (withPc)
            {
                registers = registers.Add(15);
            }
            var regList = string.Join(",", registers.Select(x => s_registerNameMap[x]));
            this.AddCode("pop", "{" + regList + "}");
        }

        public override void Move(int to, int from, string name)
        {
            this.AddCode("mov", string.Format("r{0}, r{1}", to, from), name);
        }

        public override void Pack64Bit(int to, int hi, int lo, string name)
        {
            throw new ErrorException("想定されていない命令です");
        }

        public override void Unpack64Bit(int hi, int lo, int from, string name)
        {
            throw new ErrorException("想定されていない命令です");
        }

        public override void StoreToStack(int from, int offset, string name)
        {
            this.AddCode("str",
                string.Format("r{0}, [sp, #{1}]", from, offset),
                string.Format("{0} をレジスタからスタックへ", name));
        }

        public override void LoadFromStack(int to, int offset, string name)
        {
            this.AddCode("ldr",
                string.Format("r{0}, [sp, #{1}]", to, offset),
                string.Format("{0} をスタックからレジスタへ", name));
        }

        public override void StoreToStack(int from, int offset, string name, int index)
        {
            this.AddCode("str",
                string.Format("r{0}, [sp, #{1}]", from, offset),
                string.Format("{0} の第 {1} ワードをレジスタからスタックへ", name, index));
        }

        public override void LoadFromStack(int to, int offset, string name, int index)
        {
            this.AddCode("ldr",
                string.Format("r{0}, [sp, #{1}]", to, offset),
                string.Format("{0} の第 {1} ワードをスタックからレジスタへ", name, index));
        }

        public override void Store(int from, int toBase, int offset, int size, string name, int index)
        {
            if (size <= 0)
            {
                throw new ErrorException(string.Format("{0} の転送サイズが不明です。", name));
            }
            else if (size % 4 == 0)
            {
                this.AddCode("str",
                    string.Format("r{0}, [r{1}, #{2}]", from, toBase, offset),
                    string.Format("{0} の第 {1} ワードをレジスタからメモリへ", name, index));
            }
            else
            {
                throw new ErrorException("未実装です");
            }
        }

        public override void Load(int to, int fromBase, int offset, int size, string name, int index)
        {
            if (size <= 0)
            {
                throw new ErrorException(string.Format("{0} の転送サイズが不明です。", name));
            }
            else if (size % 4 == 0)
            {
                this.AddCode("ldr",
                    string.Format("r{0}, [r{1}, #{2}]", to, fromBase, offset),
                    string.Format("{0} の第 {1} ワードをメモリからレジスタへ", name, index));
            }
            else
            {
                throw new ErrorException("未実装です");
            }
        }

        public override void LoadStackAddress(int to, int offset, string name)
        {
            this.AddCode("add",
                string.Format("r{0}, sp, #{1}", to, offset),
                string.Format("{0} のアドレスをレジスタへ", name));
        }
    }
}
