﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Text.RegularExpressions;
using System.Diagnostics;
using System.Reflection;
using System.Threading;
using System.Threading.Tasks;

namespace MakeSvcVeneer
{
    internal static class Util
    {
        public struct TextPoint
        {
            public int Lines;
            public int Chars;
        }

        public static TextPoint GetTextPoint(string s, int pos)
        {
            int lineNo = 1;
            int prevPos = 0;
            int nextPos = 0;

            for (;;)
            {
                nextPos = s.IndexOfLineHead(prevPos);

                if (nextPos < 0 || nextPos > pos)
                {
                    break;
                }
                prevPos = nextPos;
                lineNo++;
            }

            int posInLine = pos - prevPos;

            return new TextPoint() { Lines = lineNo, Chars = posInLine };
        }

        public static string FormatTextPoint(string path, TextPoint tp, int tabOffset)
        {
            if (tp.Chars >= 0 && tp.Lines >= 0)
            {
                return string.Format("line {0,3} pos {1,3} ({3,3}) file {4}| ( {5}({0},{2}) )",
                    tp.Lines,
                    tp.Chars,
                    1 + tp.Chars,
                    1 + tp.Chars + tabOffset,
                    Path.GetFileName(path),
                    Path.GetFullPath(path));
            }
            else
            {
                return string.Format(" {0}", path);
            }
        }

        public static T[] SubArray<T>(T[] a, int index, int num)
        {
            var s = new T[num];
            Array.Copy(a, index, s, 0, num);
            return s;
        }

        public static string MakeRepeatString(string one, int num)
        {
            var sb = new StringBuilder(one.Length * num);
            for (int i = 0; i < num; ++i)
            {
                sb.Append(one);
            }
            return sb.ToString();
        }

        public static void Log(string format, params object[] items)
        {
            var s = items.Length > 0 ? string.Format(format, items) : format;
            Debug.Print(s);
            Console.WriteLine(s);
        }

        public static T GetValueOrAddDefault<K, T>(Dictionary<K, T> map, K key)
            where T : new()
        {
            T item;

            if (!map.TryGetValue(key, out item))
            {
                item = new T();
                map.Add(key, item);
            }

            return item;
        }

        internal static int RoundUp(int x, int unit)
        {
            return ((x + (unit - 1)) / unit) * unit;
        }

        public static T GetValueOrDefault<K, T>(Dictionary<K, T> map, K key)
        {
            T item;
            if (!map.TryGetValue(key, out item))
            {
                item = default(T);
            }
            return item;
        }

        public static T GetValueOrDefault<K, T>(Dictionary<K, T> map, K key, T def)
        {
            T item;
            if (!map.TryGetValue(key, out item))
            {
                item = def;
            }
            return item;
        }

        public static T TrappedGetValue<K, T>(Dictionary<K, T> map, K key)
        {
            T item;
            if (map.TryGetValue(key, out item))
            {
                return item;
            }

            throw new ErrorException(
                string.Format("'{0}' が見つかりませんでした。", key));
        }

        public static string MakeRelativePath(string from, string target)
        {
            var isFromAbs = Path.IsPathRooted(from);
            var isTargetAbs = Path.IsPathRooted(target);

            if (isFromAbs == isTargetAbs)
            {
                var normalizedFrom = NormalizePath(from);
                var normalizedTarget = NormalizePath(target);

                if (isFromAbs)
                {
                    // from と target が共に絶対パスの場合
                    //   → ドライブが異なるなら target をそのまま返す
                    //      そうでなければドライブを除いて相対パスとして扱う

                    var fromRoot = Path.GetPathRoot(normalizedFrom);
                    var targetRoot = Path.GetPathRoot(normalizedTarget);

                    if (fromRoot != targetRoot)
                    {
                        return normalizedTarget;
                    }

                    normalizedFrom = normalizedFrom.Substring(fromRoot.Length);
                    normalizedTarget = normalizedTarget.Substring(targetRoot.Length);
                }

                // from と target が共に相対パスの場合
                {
                    var fromParts = normalizedFrom.Split('\\');
                    var targetParts = normalizedTarget.Split('\\');

                    var minNumParts = Math.Min(fromParts.Length, targetParts.Length);
                    int numMatch = 0;

                    for (numMatch = 0; numMatch < minNumParts; ++numMatch)
                    {
                        if (fromParts[numMatch] != targetParts[numMatch])
                        {
                            break;
                        }
                    }

                    var up = MakeRepeatString("..\\", fromParts.Length - numMatch);
                    var down = string.Join("\\", targetParts.Skip(numMatch));

                    var path = up + down;

                    return (path.Length == 0) ? "." : path;
                }
            }

            // それ以外
            //   → エラー

            throw new ErrorException("相対パスが作成できません。",
                string.Format("基点={0}\n対象={1}\n", from, target));
        }

        public static string NormalizePath(string path)
        {
            // スラッシュをバックスラッシュに
            var p1 = path.Replace('/', '\\');

            // バックスラッシュの連続を一つのバックスラッシュに
            var p2 = Regex.Replace(p1, @"\\+", @"\");

            // ドライブ文字を大文字に
            var p3 = Regex.Replace(p2, @"[a-z]:", x => x.Value.ToUpper());

            // . を消す
            // TODO: UNC の場合にローカルホストを消してしまう
            var p4 = p3.Replace(@"\.\", @"\");

            // .. を畳む
            var p5 = p4;
            {
                var dotdot = new Regex(@"([^\\]+)\\\.\.\\");
                int start = 0;

                for (;;)
                {
                    bool isMatch = false;
                    p5 = dotdot.Replace(p5,
                        (m) =>
                        {
                            isMatch = true;
                            if (m.Groups[1].Value == "..")
                            {
                                start = m.Index + 2;
                                return m.Value;
                            }
                            else
                            {
                                return string.Empty;
                            }
                        },
                        1,
                        start);

                    if (!isMatch)
                    {
                        break;
                    }
                }
            }
            if (Path.IsPathRooted(path) && !Path.IsPathRooted(p5))
            {
                var sb = new StringBuilder();
                sb.AppendFormat("パス     {0}\n", path);
                throw new ErrorException(
                    "パスに含まれる .. が多すぎます。",
                    sb.ToString());
            }

            return p5;
        }

        public static int DivUp(int v, int divider)
        {
            return (v + (divider - 1)) / divider;
        }
    }
}
