﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace MakeSvcVeneer.Parser
{
    internal class SyntaxErrorException : ApplicationException
    {
        public string[] ExpectedTokens { get; private set; }

        public SyntaxErrorException(string[] et)
        {
            this.ExpectedTokens = et;
        }
    }

    internal class SvcParserSemanticAction : ISemanticAction
    {
        private static Regex s_typeRegex = new Regex("^([0-9a-zA-Z_]+::)*([0-9a-zA-Z_]+)$");

        private Dictionary<string, TypeDeclaration> m_types = new Dictionary<string, TypeDeclaration>();

        public SvcParserSemanticAction()
        {
            foreach (var t in PredefinedTypes.List)
            {
                RegisterType(new TypeDeclarationPredefined(t.Name));
            }
        }

        public bool HasType(string key)
        {
            return m_types.ContainsKey(key);
        }
        private TypeDeclaration RegisterType(TypeDeclaration td)
        {
            Match m = s_typeRegex.Match(td.Name);
            string name = m.Groups[2].Value;

            if (!m.Success)
            {
                throw new ErrorException(
                    string.Format("型として不適切な文字列です ({0})", td.Name));
            }

            try
            {
                m_types.Add(td.Name, td);
            }
            catch (ArgumentException)
            {
                throw new ErrorException("Type '" + td.Name + "' is defined twice.");
            }

            return td;
        }

        private static T MakeList<T, U>(T list, U item)
            where T : List<U>
        {
            list.Add(item);
            return list;
        }

        private static ParameterDirectionType ParseParameterDirectionType(string s)
        {
            switch (s)
            {
            case "in": return ParameterDirectionType.In;
            case "out": return ParameterDirectionType.Out;
            case "inptr": return ParameterDirectionType.InPtr;
            case "outptr": return ParameterDirectionType.OutPtr;
            }

            throw new ErrorException(string.Format("不適切な入出力指定です ({0})", s));
        }

        #region ISemanticAction メンバー

        [System.Diagnostics.CodeAnalysis.SuppressMessage("StyleCop.CSharp.NamingRules", "SA1300:ElementMustBeginWithUpperCaseLetter", Justification = "自動生成コードが必要とするため")]
        public void syntax_error(string expected)
        {
            throw new SyntaxErrorException(expected.Split(' '));
        }

        [System.Diagnostics.CodeAnalysis.SuppressMessage("StyleCop.CSharp.NamingRules", "SA1300:ElementMustBeginWithUpperCaseLetter", Justification = "自動生成コードが必要とするため")]
        public void stack_overflow()
        {
            throw new ErrorException("internal error");
        }

        public IncludeFileList MakeList(int pos, IncludeFileList arg1, string arg2)
        {
            return MakeList(arg1, arg2);
        }
        public IncludeFileList MakeList(int pos, string arg1)
        {
            return MakeList(new IncludeFileList(), arg1);
        }
        public OperationContentList MakeList(int pos, OperationContent arg1)
        {
            return MakeList(new OperationContentList(), arg1);
        }
        public OperationContentList MakeList(int pos, OperationContentList arg1, OperationContent arg2)
        {
            return MakeList(arg1, arg2);
        }
        public OperationParameterList MakeList(int pos, OperationParameter arg1)
        {
            return MakeList(new OperationParameterList(), arg1);
        }
        public OperationParameterList MakeList(int pos, OperationParameterList arg1, OperationParameter arg2)
        {
            return MakeList(arg1, arg2);
        }
        public SvcDefinition MakeList(int pos, SvcDefinition arg1, SvcDefinitionContent arg2)
        {
            return MakeList(arg1, arg2);
        }
        public SvcDefinition MakeList(int pos, SvcDefinitionContent arg1)
        {
            return MakeList(new SvcDefinition(), arg1);
        }
        public TypeDeclarationList MakeList(int pos, TypeDeclaration arg1)
        {
            return MakeList(new TypeDeclarationList(), arg1);
        }
        public TypeDeclarationList MakeList(int pos, TypeDeclarationList arg1, TypeDeclaration arg2)
        {
            return MakeList(arg1, arg2);
        }
        public VariableDeclarationList MakeList(int pos, VariableDeclaration arg1)
        {
            return MakeList(new VariableDeclarationList(), arg1);
        }
        public VariableDeclarationList MakeList(int pos, VariableDeclarationList arg1, VariableDeclaration arg2)
        {
            return MakeList(arg1, arg2);
        }

        public TypeDeclaration Pass(int pos, TypeDeclaration arg1)
        {
            return arg1;
        }
        public OperationParameterList Pass(int pos, OperationParameterList arg1)
        {
            return arg1;
        }
        public string Pass(int pos, string arg1)
        {
            return arg1;
        }

        public Includes MakeIncludes(int pos, string arg1, IncludeFileList arg2)
        {
            return new Includes(arg1, arg2);
        }

        public OperationDeclaration MakeOperationDeclaration(int pos, OperationDeclaration arg1, OperationParameterList arg2)
        {
            return OperationDeclaration.AddParameters(arg1, arg2);
        }

        public OperationDeclaration MakeOperationDeclaration(int pos, string arg1, string arg2)
        {
            return new OperationDeclaration(arg1, arg2);
        }

        public OperationDeclaration MakeOperationDeclaration(int pos, string arg1, string arg2, int arg3)
        {
            return new OperationDeclaration(arg1, arg2, arg3);
        }

        public OperationParameter MakeOperationParameter(int pos, string arg1, VariableDeclaration arg2)
        {
            return new OperationParameter(ParseParameterDirectionType(arg1), arg2);
        }

        public OperationParameterList MakeOperationParameterList(int pos)
        {
            return new OperationParameterList();
        }

        public Operations MakeOperations(int pos, string arg1, OperationContentList arg2)
        {
            return new Operations(arg1, arg2);
        }

        public Types MakeTypes(int pos, string arg1, TypeDeclarationList arg2)
        {
            return new Types(arg1, arg2);
        }

        public VariableDeclaration MakeVariableDeclaration(int pos, string arg1, string arg2)
        {
            return new VariableDeclaration(arg1, arg2);
        }

        public int MakeOperationNumber(int pos, string arg1, int arg2)
        {
            if (arg1 != "next")
            {
                throw new ErrorException(
                    string.Format("この場所には 'next' が必要です ({0})", arg1));
            }
            return arg2;
        }

        public OperationContent MakeOperationContent(int pos, OperationDeclaration arg1)
        {
            return arg1;
        }

        public OperationContent MakeOperationContent(int pos, int arg1)
        {
            return new SetNextFunctionNumber(arg1);
        }

        public SvcDefinitionContent MakeSvcDefinitionContent(int pos, Includes arg1)
        {
            return arg1;
        }

        public SvcDefinitionContent MakeSvcDefinitionContent(int pos, Operations arg1)
        {
            return arg1;
        }

        public SvcDefinitionContent MakeSvcDefinitionContent(int pos, Types arg1)
        {
            return arg1;
        }

        public TypeDeclaration MakeAbstractTypeDeclaration(int pos, TypeDeclaration arg1)
        {
            return TypeDeclaration.SetAbstract(arg1);
        }

        public TypeDeclaration MakeAbstractCommonTypeDeclaration(int pos, TypeDeclaration arg1)
        {
            return TypeDeclaration.SetCommon(TypeDeclaration.SetAbstract(arg1));
        }

        public TypeDeclaration MakeTypeDeclarationAlias(int pos, string arg1, string arg2)
        {
            return RegisterType(new TypeDeclarationAlias(arg1, arg2));
        }

        public TypeDeclaration MakeTypeDeclarationEnum(int pos, string arg1)
        {
            return RegisterType(new TypeDeclarationEnum(arg1));
        }

        public TypeDeclaration MakeTypeDeclarationStruct(int pos, string arg1, VariableDeclarationList arg2)
        {
            return RegisterType(new TypeDeclarationStruct(arg1, arg2));
        }

        public TypeDeclaration MakeTypeDeclarationUnion(int pos, string arg1, VariableDeclarationList arg2)
        {
            return RegisterType(new TypeDeclarationUnion(arg1, arg2));
        }

        public VariableDeclaration MakeVariableDeclarationArray(int pos, string arg1, string arg2)
        {
            return VariableDeclaration.MakeArray(arg1, arg2);
        }

        public VariableDeclaration MakeVariableDeclarationArray(int pos, string arg1, string arg2, int arg3)
        {
            return VariableDeclaration.MakeArray(arg1, arg2, arg3);
        }

        public VariableDeclaration MakeVariableDeclarationArray(int pos, string arg1, string arg2, string arg3)
        {
            return VariableDeclaration.MakeArray(arg1, arg2, arg3);
        }

        #endregion
    }
}
