﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeSvcVeneer
{
    internal class Mangler
    {
        public class TypeInfo
        {
            public string Name { get; set; }
            public TypeInfo[] TemplateParameters { get; set; }
            public bool IsConst { get; set; }
            public bool IsReference { get; set; }
            public bool IsPointer { get; set; }
            public bool IsArray { get; set; }
        }
        public class FunctionInfo
        {
            public string Name { get; set; }
            public TypeInfo[] Parameters { get; set; }
        }

        private Dictionary<string, int> m_substitutionMap = new Dictionary<string, int>();
        private int m_substitutionIndex = -1;

        public static string Mangle(
            FunctionInfo fi,
            CodeGenParams cgp)
        {
            var m = new Mangler();

            var sb = new StringBuilder();

            sb.Append("_Z");
            sb.Append(m.MakeName(fi.Name, true, cgp, null));
            foreach (var p in fi.Parameters)
            {
                sb.Append(m.MakeType(p, cgp));
            }
            if (fi.Parameters.Length == 0)
            {
                sb.Append("v");
            }

            return sb.ToString();
        }

        private Mangler() { }

        private static string MakeSourceName(string name)
        {
            return string.Format("{0}{1}", name.Length, name);
        }

        private string MakeName(
            string name,
            bool isFunctionName,
            CodeGenParams cgp,
            TypeInfo[] templateParameters)
        {
            var nameParts = name.Split(new string[] { "::" }, StringSplitOptions.None);

            var replacedParts = ReplaceSubstitution(nameParts);
            if (isFunctionName)
            {
                RegisterSubstitution(nameParts.Take(nameParts.Length - 1).ToArray(), null);
            }
            else
            {
                RegisterSubstitution(nameParts, templateParameters);
            }

            var sb = new StringBuilder();

            if (replacedParts != null)
            {
                var subst = replacedParts[0];
                var unq = replacedParts.Skip(1);
                if (unq.Count() > 0 || templateParameters != null)
                {
                    var connectedName = string.Join(string.Empty, unq.Select(x => MakeSourceName(x)));

                    sb.AppendFormat("N{0}{1}", subst, connectedName);
                }
                else
                {
                    return subst;
                }
            }
            else if (nameParts.Length > 1)
            {
                var connectedName = string.Join(string.Empty, nameParts.Select(x => MakeSourceName(x)));

                sb.AppendFormat("N{0}", connectedName);
            }
            else
            {
                return string.Format("{0}{1}", name.Length, name);
            }

            if (templateParameters != null)
            {
                sb.Append("I");
                foreach (var tti in templateParameters)
                {
                    sb.Append(MakeType(tti, cgp));
                }
                sb.Append("E");
            }
            sb.Append("E");

            return sb.ToString();
        }

        private static string FixupInternalType(
            ref bool isPointer,
            string typeName,
            CodeGenParams cgp)
        {
            switch (typeName)
            {
            case "nn::Bit8":    return "uchar_t";       // unsgined char
            case "nn::Bit16":   return "ushort_t";      // unsigned short
            case "int8_t":      return "schar_t";       // signed char
            case "int16_t":     return "short_t";       // short
            case "string8":
                                isPointer = true;
                                return "char";          // char*
            case "string16":
                                isPointer = true;
                                return "char16_t";      // char16_t*
            case "uint8_t":     return "uchar_t";       // unsigned char
            case "uint16_t":    return "ushort_t";      // unsigned short
            }

            if (cgp.DataModel == CodeGenParams.DataModels.Lp64)
            {
                switch (typeName)
                {
                case "nn::Bit32":   return "uint_t";    // unsigned int
                case "nn::Bit64":   return "ulong_t";   // unsgined long
                case "int32_t":     return "int_t";     // int
                case "int64_t":     return "long_t";    // long
                case "uint32_t":    return "uint_t";    // unsigned int
                case "uint64_t":    return "ulong_t";   // unsigned long
                case "uintptr_t":   return "ulong_t";   // unsinged long
                case "size_t":      return "ulong_t";   // unsinged long
                }
            }
            else if (cgp.DataModel == CodeGenParams.DataModels.Ilp32)
            {
                switch (typeName)
                {
                case "nn::Bit32":   return "uint_t";        // unsigned int
                case "nn::Bit64":   return "ulonglong_t";   // unsgined long long
                case "int32_t":     return "int_t";         // int
                case "int64_t":     return "longlong_t";    // long long
                case "uint32_t":    return "uint_t";        // unsigned int
                case "uint64_t":    return "ulonglong_t";   // unsigned long long
                }

                if (cgp.Archtecture == CodeGenParams.ArchitectureType.AArch32)
                {
                    switch (typeName)
                    {
                    case "uintptr_t":   return "uint_t";    // unsinged int
                    case "size_t":      return "uint_t";    // unsinged int
                    }
                }
                else if (cgp.Archtecture == CodeGenParams.ArchitectureType.AArch64)
                {
                    switch (typeName)
                    {
                    case "uintptr_t":   return "ulong_t";   // unsinged long
                    case "size_t":      return "ulong_t";   // unsinged long
                    }
                }
                else
                {
                    throw new ErrorException("未知の ArchitectureType");
                }
            }
            else
            {
                throw new ErrorException("未知の DataModels");
            }

            return typeName;
        }


        private static string GetBuiltInType(
            string typeName)
        {
            switch (typeName)
            {
            case "uchar_t":     return "h";     // unsgined char
            case "ushort_t":    return "t";     // unsigned short
            case "char":        return "c";     // char
            case "char16_t":    return "Ds";    // char16_t
            case "schar_t":     return "a";     // signed char
            case "short_t":     return "s";     // short
            case "void":        return "v";     // void
            case "bool":        return "b";     // bool
            case "int_t":       return "i";     // int
            case "long_t":      return "l";     // long
            case "longlong_t":  return "x";     // long long
            case "uint_t":      return "j";     // unsigned int
            case "ulong_t":     return "m";     // unsinged long
            case "ulonglong_t": return "y";     // unsinged long long
            }

            return null;
        }

        private string MakeTypeHook(
            bool isPointer,
            bool isReference,
            bool isConst,
            bool isArray,
            string name,
            CodeGenParams cgp,
            TypeInfo[] templateParameters)
        {
            name = FixupInternalType(ref isPointer, name, cgp);
            var subst = ReplaceSubstitution(
                            MakeSubstitutionKey(
                                isPointer, isReference, isConst, isArray, name, templateParameters));
            if (subst != null)
            {
                return subst;
            }

            var ret = MakeType(
                        isPointer,
                        isReference,
                        isConst,
                        isArray,
                        name,
                        cgp,
                        templateParameters);

            // ret が 1 文字なら組み込み型として、Register しない
            if (ret.Length >= 2)
            {
                RegisterSubstitution(MakeSubstitutionKey(
                    isPointer, isReference, isConst, isArray, name, templateParameters));
            }

            return ret;
        }

        private string MakeType(
            bool isPointer,
            bool isReference,
            bool isConst,
            bool isArray,
            string name,
            CodeGenParams cgp,
            TypeInfo[] templateParameters)
        {
            if (isPointer)
            {
                return "P" + MakeTypeHook(false, isReference, isConst, isArray, name, cgp, templateParameters);
            }
            if (isReference)
            {
                return "R" + MakeTypeHook(false, false, isConst, isArray, name, cgp, templateParameters);
            }
            if (isConst)
            {
                return "K" + MakeTypeHook(false, false, false, isArray, name, cgp, templateParameters);
            }
            if (isArray)
            {
                return "A1_" + MakeTypeHook(false, false, false, false, name, cgp, templateParameters);
            }

            var builtinType = GetBuiltInType(name);
            if (builtinType != null)
            {
                return builtinType;
            }
            else
            {
                return MakeName(name, false, cgp, templateParameters);
            }
        }

        private string MakeType(TypeInfo ti, CodeGenParams cgp)
        {
            return MakeTypeHook(
                    ti.IsPointer,
                    ti.IsReference,
                    ti.IsConst,
                    ti.IsArray,
                    ti.Name,
                    cgp,
                    ti.TemplateParameters);
        }

        //----------------------------------------------------------
        // substitution

        private static string MakeSeqId(int index)
        {
            int upper = index / 36;
            int lower = index % 36;

            char c = (char)((lower < 10) ? '0' + lower : 'A' + (lower - 10));
            var s = (upper > 0) ? MakeSeqId(upper) : string.Empty;

            return s + c;
        }
        private string[] ReplaceSubstitution(string[] parts)
        {
            for (int num = parts.Length; num > 0; --num)
            {
                var key = string.Join("::", parts.Take(num));
                var subst = ReplaceSubstitution(key);
                if (subst != null)
                {
                    return new string[] { subst }.Concat(parts.Skip(num)).ToArray();
                }
            }
            return null;
        }
        private string ReplaceSubstitution(string key)
        {
            int index;
            if (m_substitutionMap.TryGetValue(key, out index))
            {
                string s = "S_";
                if (index >= 0)
                {
                    s = string.Format("S{0}_", MakeSeqId(index));
                }
                return s;
            }
            return null;
        }
        private void RegisterSubstitution(string[] parts, TypeInfo[] templateParameters)
        {
            for (int num = 1; num <= parts.Length; ++num)
            {
                var key = string.Join("::", parts.Take(num));
                RegisterSubstitution(key);
            }
        }
        private void RegisterSubstitution(string key)
        {
            if (!m_substitutionMap.ContainsKey(key))
            {
                m_substitutionMap.Add(key, m_substitutionIndex++);
            }
        }
        private string MakeSubstitutionKey(TypeInfo ti)
        {
            return MakeSubstitutionKey(ti.IsPointer,
                    ti.IsReference,
                    ti.IsConst,
                    ti.IsArray,
                    ti.Name,
                    ti.TemplateParameters);
        }
        private string MakeSubstitutionKey(
            bool isPointer,
            bool isReference,
            bool isConst,
            bool isArray,
            string name,
            TypeInfo[] templateParameters)
        {
            var sb = new StringBuilder();

            sb.Append(name);

            if (isArray)
            {
                sb.Append("[]");
            }
            if (isConst)
            {
                sb.Append(" const");
            }
            if (isPointer)
            {
                sb.Append("*");
            }
            if (isReference)
            {
                sb.Append("&");
            }
            if (templateParameters != null)
            {
                sb.AppendFormat("<{0}>",
                    string.Join(",", templateParameters.Select(x => MakeSubstitutionKey(x))));
            }

            return sb.ToString();
        }
    }
}
