﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeSvcVeneer
{
    internal class LayoutConversionOperationIndirectScatter : LayoutConversion.Operation
    {
        // ABI: スタック上のアドレスが指すメモリ
        // SVC: レジスタ x N

        public LayoutConversionOperationIndirectScatter(int abiIndex, int[] svcIndex,
            int storageSize, int transSize, string name)
            : base("IndirectScatter", abiIndex, svcIndex, storageSize, transSize, name)
        {
        }

        public override void GenerateCodeInAbiToSvc(
            CodeGenerator cg, RegisterUsageTracer tracer, int stackOffset)
        {
            var abiOffset = stackOffset + (AbiIndex * StorageSize);

            var addrRegister = SvcIndex.Last();
            tracer.Occupy(addrRegister, "svc temporary " + this.VariableName);
            cg.LoadFromStack(addrRegister, abiOffset, this.VariableName + " のアドレス");

            var index = 0;
            foreach (var si in this.SvcIndex)
            {
                var offset = index++ * StorageSize;
                tracer.Occupy(si, "svc " + this.VariableName);
                cg.Load(si, addrRegister, offset, TransferSize, this.VariableName, index);
            }
        }
        public override bool CanGenerateCodeInSvcToAbi(RegisterUsageTracer tracer)
        {
            foreach (var si in this.SvcIndex)
            {
                tracer.Release(si);
            }

            var ret = tracer.TryOccupy(AbiIndex, "abi " + VariableName);
            tracer.Release(this.AbiIndex);
            return ret;
        }
        public override void GenerateCodeInSvcToAbi(
            CodeGenerator cg, RegisterUsageTracer tracer, Dictionary<int, int> storageMap)
        {
            var storageOffset = storageMap[AbiIndex];
            int index = 0;
            foreach (var si in this.SvcIndex)
            {
                var offset = storageOffset + (index++ * StorageSize);
                tracer.Release(si);
                cg.StoreToStack(si, offset, this.VariableName, index);
            }

            tracer.Occupy(this.AbiIndex, "abi " + this.VariableName);
            cg.LoadStackAddress(this.AbiIndex, this.StorageSize, this.VariableName);
            tracer.Release(this.AbiIndex);
            cg.StoreToStack(this.SvcIndex[0], this.AbiIndex * this.StorageSize, this.VariableName + " のアドレス");
        }
        public override void GenerateCodeOutPrepareToAbi(
            CodeGenerator cg, RegisterUsageTracer tracer, Dictionary<int, int> storageMap)
        {
            var memory = storageMap[AbiIndex];
            tracer.Occupy(this.AbiIndex, "abi " + this.VariableName);
            cg.LoadStackAddress(this.SvcIndex[0], memory, this.VariableName);
            tracer.Release(this.AbiIndex);
            cg.StoreToStack(this.SvcIndex[0], this.AbiIndex * this.StorageSize, this.VariableName + " のアドレス");
        }
        public override void GenerateCodeOutAbiToSvc(
            CodeGenerator cg, RegisterUsageTracer tracer, Dictionary<int, int> storageMap)
        {
            var storageOffset = storageMap[AbiIndex];

            int index = 0;
            foreach (var si in this.SvcIndex)
            {
                var offset = storageOffset + (index++ * StorageSize);
                cg.LoadFromStack(si, offset, this.VariableName, index);
            }
        }
        public override void GenerateCodeOutSvcToAbi(
            CodeGenerator cg,
            RegisterUsageTracer tracer,
            Dictionary<int, int> saveMap,
            int safeRegister,
            int stackOffset)
        {
            var abiOffset = stackOffset + (AbiIndex * StorageSize);

            cg.LoadFromStack(safeRegister, abiOffset, this.VariableName + " のアドレス");

            int index = 0;
            foreach (var si in this.SvcIndex)
            {
                var offset = index++ * StorageSize;
                cg.Store(si, safeRegister, offset, TransferSize, this.VariableName, index);
            }
        }
    }
}
