﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeSvcVeneer
{
    internal class LayoutConversion
    {
        public abstract class Operation
        {
            // user -> svc -> handler -> return -> user
            //      ~~
            public abstract void GenerateCodeInAbiToSvc(
                CodeGenerator cg, RegisterUsageTracer tracer, int stackOffset);
            public abstract bool CanGenerateCodeInSvcToAbi(RegisterUsageTracer tracer);
            // user -> svc -> handler -> return -> user
            //             ~~
            public abstract void GenerateCodeInSvcToAbi(
                CodeGenerator cg, RegisterUsageTracer tracer, Dictionary<int, int> storageMap);
            public abstract void GenerateCodeOutPrepareToAbi(
                CodeGenerator cg, RegisterUsageTracer tracer, Dictionary<int, int> storageMap);
            // user -> svc -> handler -> return -> user
            //                        ~~
            public abstract void GenerateCodeOutAbiToSvc(
                CodeGenerator cg, RegisterUsageTracer tracer, Dictionary<int, int> storageMap);
            // user -> svc -> handler -> return -> user
            //                                  ~~
            public abstract void GenerateCodeOutSvcToAbi(
                CodeGenerator cg, RegisterUsageTracer tracer, Dictionary<int, int> saveMap, int safeRegister, int stackOffset);

            public string Type { get; private set; }

            /// <summary>
            /// この処理に関連付けられている仮引数名を取得します。
            /// </summary>
            public string VariableName { get; private set; }
            /// <summary>
            /// ABI で値が格納されているストレージ内のインデックスを取得します。
            /// ストレージは Type が Move/Scatter ならレジスタ、
            /// IndirectScatter ならスタックです。
            /// </summary>
            public int AbiIndex { get; private set; }
            /// <summary>
            /// SVC で値が格納されているストレージ内のインデックスを取得します。
            /// ストレージは常にレジスタです。
            /// 配列内の順番どおりに値が分割されて格納されます。
            /// </summary>
            public int[] SvcIndex { get; private set; }

            public int StorageSize { get; private set; }
            public int TransferSize { get; private set; }

            protected Operation(string type, int abiIndex, int[] svcIndex,
                int storageSize, int transSize, string name)
            {
                Type = type;
                AbiIndex = abiIndex;
                SvcIndex = svcIndex;
                VariableName = name;
                StorageSize = storageSize;
                TransferSize = transSize;
            }

            protected void ThrowUnexpectedOperation()
            {
                throw new ErrorException("予期しないストレージ処理です",
                    string.Format("type={0}\nname={1}", Type, VariableName));
            }
            public Operation MakeModified(int abiIndex)
            {
                var copy = (Operation)MemberwiseClone();
                copy.AbiIndex = abiIndex;
                return copy;
            }
        }

        /// <summary>
        /// ABI と SVC 間で変換処理が必要か否かを取得します。
        /// </summary>
        public bool IsRequireConversion { get; private set; }
        /// <summary>
        /// システムコール実行前に必要な変換処理のリストを取得します。
        /// </summary>
        public Operation[] PreOperations { get; private set; }
        /// <summary>
        /// システムコール実行後に必要な変換処理のリストを取得します。
        /// </summary>
        public Operation[] PostOperations { get; private set; }
        /// <summary>
        /// 最初に保存しなければならないレジスタのリストを取得します。
        /// ユニーク化され、昇順にソートされています。
        /// </summary>
        public int[] SaveRegisters { get; private set; }
        /// <summary>
        /// 最後に復帰しなければならないレジスタのリストを取得します。
        /// ユニーク化され、昇順にソートされています。
        /// </summary>
        public int[] RestoreRegisters { get; private set; }
        /// <summary>
        /// システムコール実行前後で値が維持されなければならないレジスタのリストを取得します。
        /// ユニーク化され、昇順にソートされています。
        /// </summary>
        public int[] KeepRegisters { get; private set; }
        /// <summary>
        /// システムコール実行後に破壊しても大丈夫なレジスタを取得します。
        /// </summary>
        public int SafeRegister { get; private set; }

        private LayoutConversion() { }

        public static Dictionary<string, LayoutConversion> Generate(
            Dictionary<string, AbiLayout> abi, Dictionary<string, SvcLayout> svc)
        {
            var map = new Dictionary<string, LayoutConversion>();

            foreach (var kv in abi)
            {
                map.Add(kv.Key, Generate(kv.Value, svc[kv.Key]));
            }

            return map;
        }

        private static IEnumerable<Operation> MakeMoveOperations(
            List<Layout.Position> abi,
            List<Layout.Position> svc,
            int storageSize,
            string name)
        {
            if (abi.Count != svc.Count)
            {
                // svc 32 bit x 2 → abi 64 bit x 1 のケースだけのはず
                Debug.Assert(svc.Count == 2, "assertion failed");
                Debug.Assert(abi.Count == 1, "assertion failed");

                // レジスタ → レジスタのケースだけのはず
                Debug.Assert(svc[0].Storage == Layout.StorageType.Register, "assertion failed");
                Debug.Assert(svc[1].Storage == Layout.StorageType.Register, "assertion failed");
                Debug.Assert(abi[0].Storage == Layout.StorageType.Register, "assertion failed");

                return new Operation[]
                {
                    new LayoutConversionOperationPack(
                        abi[0].Index, svc[0].Index, svc[1].Index, storageSize, name)
                };
            }
            else
            {
                return svc.Zip(abi, (s, a) => Tuple.Create(s, a))
                    // ストレージが異なるなら移動する
                    .Where(x => !x.Item1.IsSame(x.Item2))
                    .Select<Tuple<Layout.Position, Layout.Position>, Operation>(pair =>
                    {
                        Debug.Assert(pair.Item1.Storage == Layout.StorageType.Register, "assertion failed");

                        if (pair.Item2.Storage == Layout.StorageType.Register)
                        {
                            return new LayoutConversionOperationMove(
                                pair.Item2.Index, pair.Item1.Index, storageSize, name);
                        }
                        else
                        {
                            return new LayoutConversionOperationLoad(
                                pair.Item2.Index, pair.Item1.Index, storageSize, name);
                        }
                    });
            }
        }
        private static Operation MakeScatterOperation(
            List<Layout.Position> abi,
            List<Layout.Position> svc,
            int storageSize,
            int transferSize,
            string name)
        {
            Debug.Assert(abi.Count == 1, "assertion failed");
            var abi0 = abi.First();
            var svcList = svc.Select(x => x.Index).ToArray();

            if (abi0.Storage == Layout.StorageType.Stack)
            {
                return new LayoutConversionOperationIndirectScatter(
                    abi0.Index, svcList, storageSize, transferSize, name);
            }
            else
            {
                return new LayoutConversionOperationScatter(
                    abi0.Index, svcList, storageSize, transferSize, name);
            }
        }
        private static Operation[] GenerateOperations(AbiLayout abi, Layout svcLayout)
        {
            var operations = new List<Operation>();

            foreach (var se in svcLayout.Params)
            {
                var svcEntry = se;
                var abiEntry = abi.GetEntry(se.VariableName);

                if (svcEntry.IsReference == abiEntry.IsReference)
                {
                    // 参照→参照、値→値 の場合

                    operations.AddRange(
                        MakeMoveOperations(
                            abiEntry.Positions,
                            svcEntry.Positions,
                            abi.CodeGenParams.StorageSize,
                            svcEntry.VariableName));
                }
                else
                {
                    // 参照→値 の場合

                    operations.Add(
                        MakeScatterOperation(
                            abiEntry.Positions,
                            svcEntry.Positions,
                            svcLayout.StorageSize,
                            abiEntry.VariableDataSize,
                            svcEntry.VariableName));
                }
            }

            return operations.ToArray();
        }

        private static int[] CollectKeepRegisters(SvcLayout svc, int abiVolatileRegisterCount)
        {
            var keeps = new List<int>();
            int volatileCount = Math.Min(abiVolatileRegisterCount, svc.CodeGenParams.RegisterCount);

            for (int i = svc.CodeGenParams.VolatileRegisterCount; i < volatileCount; ++i)
            {
                if (svc.In.IsRegisterAvailable(i) && svc.Out.IsRegisterAvailable(i))
                {
                    keeps.Add(i);
                }
            }

            return keeps.ToArray();
        }

        private static int FindSafeRegister(out bool isRequireSave, AbiLayout abi, SvcLayout svc)
        {
            // 候補レジスタ: abi の揮発レジスタのうち out で使用されていないもの
            // 使用できないレジスタ: svc の out で使用されているレジスタ
            for (int i = 0; i < abi.In.RegisterCount; ++i)
            {
                if (abi.Out.IsRegisterAvailable(i)
                    && svc.Out.IsRegisterAvailable(i))
                {
                    isRequireSave = false;
                    return i;
                }
            }

            // 見つからなかったら
            // abi の不揮発レジスタのうち svc の out で使用されていないレジスタを使う
            // この場合 save と restore が必要
            for (int i = abi.In.RegisterCount; i < svc.Out.RegisterCount + 1; ++i)
            {
                if (svc.Out.IsRegisterAvailable(i))
                {
                    isRequireSave = true;
                    return i;
                }
            }
            throw new ErrorException("内部エラー");
        }

        private static LayoutConversion Generate(AbiLayout abi, SvcLayout svc)
        {
            // in の変換
            var preOperations = GenerateOperations(abi, svc.In);

            // out の変換
            var postOperations = GenerateOperations(abi, svc.Out);

            // post の scatter で参照するレジスタの値は保存しなければならない
            var saveRegisters = postOperations.Where(x => x is LayoutConversionOperationScatter)
                                              .Select(x => x.AbiIndex)
                                              .ToList();

            // svc の in/out で使用する不揮発レジスタは保存しなければならない
            for (int i = abi.In.RegisterCount; i < svc.In.RegisterCount; ++i)
            {
                if ((!svc.In.IsRegisterAvailable(i))
                    || (!svc.Out.IsRegisterAvailable(i)))
                {
                    saveRegisters.Add(i);
                }
            }

            // out 時に一時的に使用できるレジスタを見つけておく
            bool isRequireSave;
            int safeRegister = FindSafeRegister(out isRequireSave, abi, svc);

            if (isRequireSave)
            {
                saveRegisters.Add(safeRegister);
            }

            // sort and unique
            saveRegisters = saveRegisters.ToLookup(x => x).Select(x => x.Key).OrderBy(x => x).ToList();

            var keepRegisters = CollectKeepRegisters(svc, abi.CodeGenParams.VolatileRegisterCount);

            var lc = new LayoutConversion();
            lc.IsRequireConversion  = preOperations.HasItem() || postOperations.HasItem() || keepRegisters.HasItem();
            lc.PreOperations        = preOperations;
            lc.PostOperations       = postOperations;
            lc.SaveRegisters        = saveRegisters.ToArray();
            lc.RestoreRegisters     = saveRegisters.Where(x => x >= abi.In.RegisterCount).ToArray();
            lc.KeepRegisters        = keepRegisters;
            lc.SafeRegister         = safeRegister;
            return lc;
        }
    }
}
