﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeSvcVeneer
{
    internal class AArch64CodeGenerator : CodeGenerator
    {
        private bool m_Is32BitStruct;

        public AArch64CodeGenerator(bool is32BitStruct)
        {
            // SVC が 32 bit のケース
            m_Is32BitStruct = is32BitStruct;
        }

        private string RegisterPrefixForStruct
        {
            get
            {
                return m_Is32BitStruct ? "w" : "x";
            }
        }

        public override void CallSystem(string id)
        {
            this.AddCode("svc", string.Format("#{0}", id));
        }

        public override void CallFunction(string name)
        {
            this.AddCode("bl", name);
        }

        public override void Return()
        {
            this.AddCode("ret", string.Empty);
        }

        public override void AllocateFromStack(int size)
        {
            this.AddCode("sub", string.Format("sp, sp, #{0}", (size + 15) / 16 * 16));
        }

        public override void FreeToStack(int size)
        {
            this.AddCode("add", string.Format("sp, sp, #{0}", (size + 15) / 16 * 16));
        }

        public override void SaveRegisters(int[] regs, bool withLr, bool isRequirePadding)
        {
            var registers = new List<int>(regs);
            if (withLr)
            {
                if ((registers.Count % 2) == 0)
                {
                    registers.Add(29);
                }
                registers.Add(30);
            }

            int num = registers.Count;

            if ((num % 2) == 1)
            {
                this.AddCode("str", string.Format("x{0}, [sp, #-16]!", registers[num - 1]));
            }

            for (int i = num / 2 - 1; i >= 0; i--)
            {
                this.AddCode("stp", string.Format("x{0}, x{1}, [sp, #-16]!", registers[i * 2], registers[i * 2 + 1]));
            }
        }

        public override void RestoreRegisters(int[] regs, bool withPc, bool isRequirePadding)
        {
            var registers = new List<int>(regs);
            if (withPc)
            {
                if ((registers.Count % 2) == 0)
                {
                    registers.Add(29);
                }
                registers.Add(30);
            }
            int num = registers.Count;

            for (int i = 0; i <= num / 2 - 1; i++)
            {
                this.AddCode("ldp", string.Format("x{0}, x{1}, [sp], #16", registers[i * 2], registers[i * 2 + 1]));
            }

            if ((num % 2) == 1)
            {
                this.AddCode("ldr", string.Format("x{0}, [sp], #16", registers[num - 1]));
            }

            if (withPc)
            {
                Return();
            }
        }

        public override void Move(int to, int from, string name)
        {
            this.AddCode("mov", string.Format("x{0}, x{1}", to, from), name);
        }

        public override void Pack64Bit(int to, int hi, int lo, string name)
        {
            this.AddCode("orr",
                string.Format("x{0}, x{1}, x{2}, lsl #32", to, lo, hi),
                string.Format("{0} を 32 bit x2 から 64 bit へ", name));
        }

        public override void Unpack64Bit(int hi, int lo, int from, string name)
        {
            if (lo != from)
            {
                this.Move(lo, from, string.Format("{0} の Lo", name));
            }

            this.AddCode("lsr", string.Format("x{0}, x{1}, #32", hi, from), string.Format("{0} の Hi", name));
        }

        public override void StoreToStack(int from, int offset, string name)
        {
            this.AddCode("str",
                string.Format("x{0}, [sp, #{1}]", from, offset),
                string.Format("{0} をレジスタからスタックへ", name));
        }

        public override void LoadFromStack(int to, int offset, string name)
        {
            this.AddCode("ldr",
                string.Format("x{0}, [sp, #{1}]", to, offset),
                string.Format("{0} をスタックからレジスタへ", name));
        }

        public override void StoreToStack(int from, int offset, string name, int index)
        {
            this.AddCode("str",
                string.Format("{0}{1}, [sp, #{2}]", RegisterPrefixForStruct, from, offset),
                string.Format("{0} の第 {1} ワードをレジスタからスタックへ", name, index));
        }

        public override void LoadFromStack(int to, int offset, string name, int index)
        {
            this.AddCode("ldr",
                string.Format("{0}{1}, [sp, #{2}]", RegisterPrefixForStruct, to, offset),
                string.Format("{0} の第 {1} ワードをスタックからレジスタへ", name, index));
        }

        public override void Store(int from, int toBase, int offset, int size, string name, int index)
        {
            if (size <= 0)
            {
                throw new ErrorException(string.Format("{0} の転送サイズが不明です。", name));
            }
            else if (size % 8 == 0)
            {
                this.AddCode("str",
                    string.Format("x{0}, [x{1}, #{2}]", from, toBase, offset),
                    string.Format("{0} の第 {1} ワードをレジスタからメモリへ", name, index));
            }
            else if (size == 4)
            {
                this.AddCode("str",
                    string.Format("w{0}, [x{1}, #{2}]", from, toBase, offset),
                    string.Format("{0} の第 {1} ワードをレジスタからメモリへ", name, index));
            }
            else
            {
                throw new ErrorException("未実装です");
            }
        }

        public override void Load(int to, int fromBase, int offset, int size, string name, int index)
        {
            if (size <= 0)
            {
                throw new ErrorException(string.Format("{0} の転送サイズが不明です。", name));
            }
            else if (size % 8 == 0)
            {
                this.AddCode("ldr",
                    string.Format("x{0}, [x{1}, #{2}]", to, fromBase, offset),
                    string.Format("{0} の第 {1} ワードをメモリからレジスタへ", name, index));
            }
            else if (size == 4)
            {
                this.AddCode("ldr",
                    string.Format("w{0}, [x{1}, #{2}]", to, fromBase, offset),
                    string.Format("{0} の第 {1} ワードをメモリからレジスタへ", name, index));
            }
            else
            {
                throw new ErrorException("未実装です");
            }
        }

        public override void LoadStackAddress(int to, int offset, string name)
        {
            this.AddCode("add",
                string.Format("x{0}, sp, #{1}", to, offset),
                string.Format("{0} のアドレスをレジスタへ", name));
        }
    }
}
