﻿using System;
using System.Diagnostics;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using Nintendo.Foundation.IO;

namespace Nintendo.MakeSdkLogHeader
{
    internal class CommandArgument
    {
        [CommandLineOption('m', "module", IsRequired = true, Description = "Target module name.")]
        public string Module { get; set; }

        [CommandLineOption('o', "output", IsRequired = true, Description = "Output file name.")]
        public string Output { get; set; }
    }

    internal class CommandInterface
    {
        public static void CreateSdkLogHeader(string moduleName, string outputPath)
        {
            using (var sw = new StreamWriter(outputPath, false, new UTF8Encoding(true)))
            {
                sw.WriteLine(
                    Properties.Resources.HeaderString.Replace(
                        "${UPPERCASE_MODULE_NAME}", moduleName.ToUpper()).Replace(
                        "${LOWERCASE_MODULE_NAME}", moduleName.ToLower()).AlignPadding(
                            @"NN_DETAIL_STRUCTURED_SDK_"));
            }
        }

        public static int Run(string[] args)
        {
            Thread.CurrentThread.CurrentUICulture = new CultureInfo("en", true);

            try
            {
                var settings = new CommandLineParserSettings()
                {
                    ApplicationDescription = "Create log API header for SDK module."
                };

                var parser = new CommandLineParser(settings);

                CommandArgument parsed;
                if (false == parser.ParseArgs(args, out parsed))
                {
                    return 1;
                }

                CreateSdkLogHeader(parsed.Module, parsed.Output);
                return 0;
            }
            catch (Exception exception)
            {
                PrintException(exception);
                return 1;
            }
        }

        public static void PrintException(Exception exception)
        {
            Console.Error.WriteLine("[ERROR] {0}", exception.Message);
            Trace.WriteLine(string.Format("StackTrace: {0}", exception.StackTrace));
        }
    }

    internal static class StringExtention
    {
        public static string AlignPadding(this string text, string paddingTarget)
        {
            var r = new Regex($@"(?<head>.+)\s+(?<tail>{paddingTarget})");
            var ms = r.Matches(text);
            var longestHead = (from Match m in ms select m.Groups["head"].Value.Length).Max();
            var columnPosition = longestHead + 4 - (longestHead % 4);
            var sb = new StringBuilder(text);
            foreach (var m in (from Match m in ms select m).Reverse())
            {
                var paddedText = m.Groups["head"].Value.PadRight(columnPosition) + m.Groups["tail"].Value;
                sb.Remove(m.Index, m.Length).Insert(m.Index, paddedText);
            }
            return sb.ToString();
        }
    }
}
