﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;

namespace MakeNso
{
    static class Lz4
    {
        private class ScopedGCHandle : IDisposable
        {
            private bool disposed;
            private GCHandle gchandle;

            public ScopedGCHandle(object value)
            {
                this.gchandle = GCHandle.Alloc(value);
            }

            public ScopedGCHandle(object value, GCHandleType type)
            {
                this.gchandle = GCHandle.Alloc(value, type);
            }

            ~ScopedGCHandle()
            {
                this.Dispose(false);
            }

            public static implicit operator IntPtr(ScopedGCHandle scopedGCHandle)
            {
                return (IntPtr)scopedGCHandle;
            }

            public void Dispose()
            {
                this.Dispose(true);
                GC.SuppressFinalize(this);
            }

            protected virtual void Dispose(bool disposing)
            {
                if (!this.disposed)
                {
                    if (disposing)
                    {
                        // ここでマネージドリソースをDisposeする
                    }

                    // ここでアンマネージドリソースを解放する
                    this.gchandle.Free();
                    this.disposed = true;
                }
            }
        }

        public static int LZ4_compress_default(byte[] source, byte[] dest, int sourceSize, int maxDestSize)
        {
            using (var gcSourceHandle = new ScopedGCHandle(source, GCHandleType.Pinned))
            using (var gcDestHandle = new ScopedGCHandle(dest, GCHandleType.Pinned))
            {
                return LZ4_compress_default(
                    Marshal.UnsafeAddrOfPinnedArrayElement(source, 0),
                    Marshal.UnsafeAddrOfPinnedArrayElement(dest, 0),
                    sourceSize,
                    maxDestSize);
            }
        }

        public static int LZ4_decompress_safe(byte[] source, int sourceOffset, byte[] dest, int destOffset, int compressedSize, int maxDecompressedSize)
        {
            using (var gcSourceHandle = new ScopedGCHandle(source, GCHandleType.Pinned))
            using (var gcDestHandle = new ScopedGCHandle(dest, GCHandleType.Pinned))
            {
                return LZ4_decompress_safe(
                    Marshal.UnsafeAddrOfPinnedArrayElement(source, sourceOffset),
                    Marshal.UnsafeAddrOfPinnedArrayElement(dest, destOffset),
                    compressedSize,
                    maxDecompressedSize);
            }
        }

        [DllImport("MakeNso.lz4.dll", CallingConvention = CallingConvention.Cdecl)]
        private static extern int LZ4_compress_default(IntPtr source, IntPtr dest, int sourceSize, int maxDestSize);

        [DllImport("MakeNso.lz4.dll", CallingConvention = CallingConvention.Cdecl)]
        private static extern int LZ4_decompress_safe(IntPtr source, IntPtr dest, int compressedSize, int maxDecompressedSize);

        [DllImport("MakeNso.lz4.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern int LZ4_compressBound(int inputSize);
    }
}
