﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace MakeNso.Elf
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;

    /// <summary>
    /// ELF データを表します。
    /// </summary>
    public sealed class ElfInfo
    {
        /// <summary>
        /// ElfInfo クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="fileName">ELF ファイルのパスです。</param>
        public ElfInfo(string fileName)
        {
            this.FileName = fileName;

            using (var fs = File.OpenRead(fileName))
            using (var er = new ElfBinaryReader(fs))
            {
                if (!IsValidElfMagic(er.ReadBytes(4)))
                {
                    throw new ArgumentException(Properties.Resources.Message_InvalidElfMagic);
                }

                this.Architecture = ConvertToElfArchitectureType(er.ReadByte());

                this.ByteOrder = ConvertToElfByteOrderType(er.ReadByte());

                er.ReadByte(); // specification version

                er.ReadByte(); // ABI type

                er.ReadByte(); // ABI version

                er.ReadBytes(7); // padding

                er.Architecture = this.Architecture;

                er.ByteOrder = this.ByteOrder;

                this.FileType = ConvertToElfFileType(er.ReadUInt16());

                this.Machine = ConvertToElfMachineType(er.ReadUInt16());

                er.ReadUInt32(); // format version

                er.ReadWord(); // entry point address

                this.ProgramHeaderTableOffset = er.ReadWord();

                this.SectionHeaderTableOffset = er.ReadWord();

                er.ReadUInt32(); // processor flags

                if (!IsValidElfHeaderSize(er.ReadUInt16(), this.Architecture))
                {
                    throw new ArgumentException(Properties.Resources.Message_InvalidElfHeaderSize);
                }

                this.ProgramHeaderEntrySize = er.ReadUInt16();

                this.NumberOfProgramHeaderEntries = er.ReadUInt16();

                this.SectionHeaderEntrySize = er.ReadUInt16();

                this.NumberOfSectionHeaderEntries = er.ReadUInt16();

                this.StringTableIndex = er.ReadUInt16();
            }

            this.SectionInfos = ElfSectionInfo.GetElfSectionInfos(
                this,
                this.SectionHeaderTableOffset,
                this.SectionHeaderEntrySize,
                this.NumberOfSectionHeaderEntries,
                this.StringTableIndex);

            this.SegmentInfos = ElfSegmentInfo.GetElfSegmentInfos(
                this,
                this.ProgramHeaderTableOffset,
                this.ProgramHeaderEntrySize,
                this.NumberOfProgramHeaderEntries,
                this.SectionInfos);

            if (this.SegmentInfos.Where(segment => segment.SegmentType == ElfSegmentType.LOAD
                    && !((segment.IsReadable() && !segment.IsWritable() && segment.IsExecutable())
                        || (segment.IsReadable() && !segment.IsWritable() && !segment.IsExecutable())
                        || (segment.IsReadable() && segment.IsWritable() && !segment.IsExecutable()))
                    && (segment.MemorySize > 0 || segment.SegmentSize > 0)).Count() > 0)
            {
                throw new ArgumentException(Properties.Resources.Message_InvalidSegment);
            }
        }

        /// <summary>
        /// ELF ファイルのパスを取得します。
        /// </summary>
        public string FileName { get; private set; }

        /// <summary>
        /// アーキテクチャタイプを取得します。
        /// </summary>
        public ElfArchitectureType Architecture { get; private set; }

        /// <summary>
        /// バイトオーダタイプを取得します。
        /// </summary>
        public ElfByteOrderType ByteOrder { get; private set; }

        /// <summary>
        /// ファイルタイプを取得します。
        /// </summary>
        public ElfFileType FileType { get; private set; }

        /// <summary>
        /// マシンタイプを取得します。
        /// </summary>
        public ElfMachineType Machine { get; private set; }

        /// <summary>
        /// ELF に属するセクションを取得します。
        /// </summary>
        public IReadOnlyList<ElfSectionInfo> SectionInfos { get; private set; }

        private ulong ProgramHeaderTableOffset { get; set; }

        private ulong SectionHeaderTableOffset { get; set; }

        private ushort ProgramHeaderEntrySize { get; set; }

        private ushort NumberOfProgramHeaderEntries { get; set; }

        private ushort SectionHeaderEntrySize { get; set; }

        private ushort NumberOfSectionHeaderEntries { get; set; }

        private ushort StringTableIndex { get; set; }

        private IReadOnlyList<ElfSegmentInfo> SegmentInfos { get; set; }

        /// <summary>
        /// api_infoセクションのセクション情報を取得します。
        /// </summary>
        /// <returns>api_infoセクションのセクション情報です。</returns>
        public ElfSectionInfo GetApiInfoElfSectionInfo()
        {
            foreach (var section in SectionInfos)
            {
                if (section.SectionName == ".api_info")
                {
                    return section;
                }
            }
            return null;
        }
        /// <summary>
        /// dynstrセクションのセクション情報を取得します。
        /// </summary>
        /// <returns>dynstrセクションのセクション情報です。</returns>
        public ElfSectionInfo GetDynStrElfSectionInfo()
        {
            foreach (var section in SectionInfos)
            {
                if (section.SectionName == ".dynstr")
                {
                    return section;
                }
            }
            return null;
        }
        /// <summary>
        /// dynsymセクションのセクション情報を取得します。
        /// </summary>
        /// <returns>dynsymセクションのセクション情報です。</returns>
        public ElfSectionInfo GetDynSymElfSectionInfo()
        {
            foreach (var section in SectionInfos)
            {
                if (section.SectionName == ".dynsym")
                {
                    return section;
                }
            }
            return null;
        }
        /// <summary>
        /// EX バイナリとして使用可能なセグメントを取得します。
        /// </summary>
        /// <returns>EX バイナリとして使用可能なセグメントです。</returns>
        public ElfSegmentInfo GetExElfSegmentInfo()
        {
            ElfSegmentInfo exSegment = null;
            foreach (var segment in SegmentInfos)
            {
                if (segment.SegmentType == ElfSegmentType.LOAD)
                {
                    if (segment.IsReadable() && !segment.IsWritable() && segment.IsExecutable())
                    {
                        if (exSegment != null)
                        {
                            throw new ArgumentException(Properties.Resources.Message_InvalidExSegment);
                        }
                        exSegment = segment;
                    }
                }
            }
            return exSegment;
        }

        /// <summary>
        /// RO バイナリとして使用可能なセグメントを取得します。
        /// </summary>
        /// <returns>RO バイナリとして使用可能なセグメントです。セグメントが見つからなかった場合は null を返します。</returns>
        public ElfSegmentInfo GetRoElfSegmentInfo()
        {
            ElfSegmentInfo roSegment = null;
            foreach (var segment in SegmentInfos)
            {
                if (segment.SegmentType == ElfSegmentType.LOAD)
                {
                    if (segment.IsReadable() && !segment.IsWritable() && !segment.IsExecutable())
                    {
                        if (roSegment != null)
                        {
                            throw new ArgumentException(Properties.Resources.Message_InvalidRoSegment);
                        }
                        roSegment = segment;
                    }
                }
            }
            return roSegment;
        }

        /// <summary>
        /// RW バイナリとして使用可能なセグメントを取得します。
        /// </summary>
        /// <returns>RW バイナリとして使用可能なセグメントです。セグメントが見つからなかった場合は null を返します。</returns>
        public ElfSegmentInfo GetRwElfSegmentInfo()
        {
            ElfSegmentInfo rwSegment = null;
            foreach (var segment in SegmentInfos)
            {
                if (segment.SegmentType == ElfSegmentType.LOAD)
                {
                    if (segment.IsReadable() && segment.IsWritable() && !segment.IsExecutable() && segment.SegmentSize > 0)
                    {
                        if (rwSegment != null)
                        {
                            throw new ArgumentException(Properties.Resources.Message_InvalidRwSegment);
                        }
                        rwSegment = segment;
                    }
                }
            }
            return rwSegment;
        }

        /// <summary>
        /// ZI 領域として使用可能なセグメントを取得します。
        /// </summary>
        /// <returns>ZI 領域として使用可能なセグメントです。セグメントが見つからなかった場合は null を返します。</returns>
        public ElfSegmentInfo GetZiElfSegmentInfo()
        {
            ElfSegmentInfo ziSegment = null;
            foreach (var segment in SegmentInfos)
            {
                if (segment.SegmentType == ElfSegmentType.LOAD)
                {
                    if (segment.IsReadable() && segment.IsWritable() && !segment.IsExecutable() && segment.SegmentSize == 0)
                    {
                        if (ziSegment != null)
                        {
                            throw new ArgumentException(Properties.Resources.Message_InvalidRwSegment);
                        }
                        ziSegment = segment;
                    }
                }
            }
            return ziSegment;
        }

        /// <summary>
        /// 現在のオブジェクトを表す文字列を返します。
        /// </summary>
        /// <returns>
        /// 現在のオブジェクトを表す文字列です。
        /// </returns>
        public override string ToString()
        {
            var sb = new StringBuilder();

            sb.AppendLine("ELF Information:");

            sb.AppendLine(string.Format("  FileName: {0}", Path.GetFileName(this.FileName)));

            sb.AppendLine(string.Format("  Class:    {0}", this.Architecture));

            sb.AppendLine(string.Format("  Data:     {0}", this.ByteOrder));

            sb.AppendLine(string.Format("  Machine:  {0}", this.Machine));

            sb.Append("  Segments: ");

            if (this.SegmentInfos.Count() == 0)
            {
                sb.Append("No Segments");
            }
            else
            {
                sb.AppendLine();

                sb.AppendFormat("    {0,-28} {1,-14} {2,-10} {3,-7} {4,-7} {5,-3}", "Name", "Type", "VirtAddr", "FileSiz", "MemSiz", "Flg");

                foreach (var segmentInfo in this.SegmentInfos)
                {
                    sb.AppendLine();

                    sb.AppendFormat("    ");

                    sb.AppendFormat("{0,-28} ", string.Join(",", segmentInfo.SegmentNames));

                    sb.AppendFormat("{0,-14} ", segmentInfo.SegmentType);

                    sb.AppendFormat("0x{0} ", segmentInfo.VirtualAddress.ToString("x8"));

                    sb.AppendFormat("0x{0} ", segmentInfo.SegmentSize.ToString("x5"));

                    sb.AppendFormat("0x{0} ", segmentInfo.MemorySize.ToString("x5"));

                    sb.Append(segmentInfo.IsReadable() ? "R" : " ");

                    sb.Append(segmentInfo.IsWritable() ? "W" : " ");

                    sb.Append(segmentInfo.IsExecutable() ? "E" : " ");
                }
            }

            return sb.ToString();
        }

        private static bool IsValidElfMagic(byte[] bytes)
        {
            var magic = new byte[] { 0x7F, (byte)'E', (byte)'L', (byte)'F' };

            return bytes.SequenceEqual(magic);
        }

        private static ElfArchitectureType ConvertToElfArchitectureType(byte value)
        {
            switch (value)
            {
                case 1: return ElfArchitectureType.Elf32;
                case 2: return ElfArchitectureType.Elf64;
                default: throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidElfArchitecture, value));
            }
        }

        private static ElfByteOrderType ConvertToElfByteOrderType(byte value)
        {
            switch (value)
            {
                case 1: return ElfByteOrderType.LittleEndian;
                case 2: return ElfByteOrderType.BigEndian;
                default: throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidElfByteOrder, value));
            }
        }

        private static ElfMachineType ConvertToElfMachineType(ushort value)
        {
            switch (value)
            {
                case 20: return ElfMachineType.PowerPC;
                case 40: return ElfMachineType.ARM;
                case 183: return ElfMachineType.AArch64;
                default: return ElfMachineType.Unknown;
            }
        }

        private static ElfFileType ConvertToElfFileType(ushort value)
        {
            switch (value)
            {
                case 0: return ElfFileType.None;
                case 1: return ElfFileType.RelocatableFile;
                case 2: return ElfFileType.ExcutableFile;
                case 3: return ElfFileType.SharedObjectFile;
                case 4: return ElfFileType.CoreFile;
                case 0xFE00: return ElfFileType.LoOs;
                case 0xFEFF: return ElfFileType.HiOs;
                case 0xff00: return ElfFileType.LoProc;
                case 0xffff: return ElfFileType.HiProc;
                default: throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidElfFileType, value));
            }
        }

        private static bool IsValidElfHeaderSize(ushort size, ElfArchitectureType architecture)
        {
            if (architecture == ElfArchitectureType.Elf32 && size == 52)
            {
                return true;
            }

            if (architecture == ElfArchitectureType.Elf64 && size == 64)
            {
                return true;
            }

            return false;
        }
    }
}
