﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Text;
using System.IO;
using System.Linq;
using System.Collections.Generic;
using MakeNro.Elf;

namespace MakeNro
{
    internal class Program
    {
        private static void Main(string[] args)
        {
            if (Directory.Exists(Path.Combine(Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location), "en")))
            {
                System.Threading.Thread.CurrentThread.CurrentUICulture = new System.Globalization.CultureInfo("en", false);
            }

            MakeNroArgs opt = new MakeNroArgs();

            // コマンドライン引数のパースはパース処理の中で、エラーを表示するため、try を分ける
            try
            {
                if (!opt.ParseArgs(args))
                {
                    return;
                }
            }
            catch
            {
                Environment.ExitCode = 1;
                return;
            }

            try
            {
                MakeNroParams param = opt.Params;
                NroFile nroFile = new NroFile();

                ElfInfo elf = new ElfInfo(param.DsoFileName);

                foreach (var section in elf.SectionInfos)
                {
                    if (elf.FileType == ElfFileType.SharedObjectFile && section.AddressAlign > 0x1000)
                    {
                        throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidSectionAlign, section.SectionName, section.AddressAlign));
                    }
                }

                byte[] buildId = BuildId.GetBuildId(elf);
                if (buildId != null)
                {
                    nroFile.SetModuleId(buildId);
                }

                ElfSegmentInfo textInfo = elf.GetExElfSegmentInfo();
                ElfSegmentInfo roInfo = elf.GetRoElfSegmentInfo();
                ElfSegmentInfo dataInfo = elf.GetRwElfSegmentInfo();
                ElfSegmentInfo bssInfo = elf.GetZiElfSegmentInfo();

                nroFile.SetTextSegment(elf.GetExElfSegmentInfo());
                nroFile.SetRoSegment(elf.GetRoElfSegmentInfo());
                nroFile.SetDataSegment(elf.GetRwElfSegmentInfo());
                nroFile.SetBssSegment(elf.GetZiElfSegmentInfo());

                ElfSectionInfo apiInfo = elf.GetApiInfoElfSectionInfo();
                if (apiInfo != null)
                {
                    nroFile.SetApiInfo(apiInfo.VirtualAddress - roInfo.VirtualAddress, apiInfo.SectionSize);
                }

                nroFile.CalcPosition();

                ElfSectionInfo dynStrInfo = elf.GetDynStrElfSectionInfo();
                if (dynStrInfo != null)
                {
                    nroFile.SetDynStrInfo(dynStrInfo.VirtualAddress - roInfo.VirtualAddress, dynStrInfo.SectionSize);
                }

                ElfSectionInfo dynSymInfo = elf.GetDynSymElfSectionInfo();
                if (dynSymInfo != null)
                {
                    nroFile.SetDynSymInfo(dynSymInfo.VirtualAddress - roInfo.VirtualAddress, dynSymInfo.SectionSize);
                }

                /// DsoHandleOffset に GetDsoHandle 関数のオフセットをセットする。
                ulong dsoHandleAddress = elf.GetDsoHandleOffset();
                nroFile.SetDsoHandleOffset(dsoHandleAddress);

                using (FileStream outputFs = new FileStream(param.NroFileName, FileMode.Create, FileAccess.Write))
                {
                    nroFile.WriteData(outputFs);
                }

                if (param.VerboseMode)
                {
                    nroFile.PrintNroHeader();
                }
            }
            catch (Exception ex)
            {
                Console.Error.WriteLine(string.Format("MakeNro INPUT='{0}', OUTPUT='{1}'", opt.Params.DsoFileName, opt.Params.NroFileName));
                Console.Error.WriteLine(ex.Message);
                Environment.ExitCode = 1;
                return;
            }
        }
    }
}
