﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Xml;
using System.Xml.Serialization;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeMeta
{
    public class Program
    {
        public static void Main(string[] args)
        {
            if (Directory.Exists(Path.Combine( Path.GetDirectoryName( System.Reflection.Assembly.GetExecutingAssembly().Location ), "en" )))
            {
                System.Threading.Thread.CurrentThread.CurrentUICulture = new System.Globalization.CultureInfo("en", false);
            }

            MakeMetaArgs opt = new MakeMetaArgs();
            try
            {
                if (!opt.ParseArgs(args))
                {
                    return;
                }
            }
            catch
            {
                Environment.ExitCode = 1;
                return;
            }
            MakeMetaParams param = opt.Params;

            OptionVariable[] variables = null;
            if (param.OptionVariables != null)
            {
                variables = new OptionVariable[param.OptionVariables.Count()];
                for (int i = 0; i < param.OptionVariables.Count(); i++)
                {
                    variables[i] = new OptionVariable(param.OptionVariables[i]);
                }
            }

            try
            {
                NpdmFile npdmFile = new NpdmFile();

                DescModel.OutputDescModel descModel;
                using (FileStream fs = new FileStream(param.InputDescPath, FileMode.Open, FileAccess.Read))
                {
                    System.Xml.Serialization.XmlSerializer serializer = new System.Xml.Serialization.XmlSerializer(typeof(DescModel.OutputDescModel));
                    serializer.UnknownElement += serializer_UnknownElement;

                    descModel = (DescModel.OutputDescModel)serializer.Deserialize(fs);
                }

                MetaModel.MetaModel metaModel;
                using (FileStream fs = new FileStream(param.InputMetaPath, FileMode.Open, FileAccess.Read))
                {
                    StreamReader sr = new StreamReader(fs);
                    bool isNewMetaRootName = false;
                    using (MemoryStream tmpMs = new MemoryStream())
                    {
                        StreamWriter sw = new StreamWriter(tmpMs);
                        while(!sr.EndOfStream)
                        {
                            string line = sr.ReadLine();
                            if (line.IndexOf("<NintendoSdkMeta>") >= 0)
                            {
                                isNewMetaRootName = true;
                            }
                            if (variables != null)
                            {
                                foreach (var variable in variables)
                                {
                                    if (line.IndexOf(variable.VariableLine) >= 0)
                                    {
                                        line = line.Replace(variable.VariableLine, variable.ValueLine);
                                    }
                                }
                            }
                            sw.WriteLine(line);
                        }
                        sw.Flush();

                        var modelType = isNewMetaRootName ? typeof(MetaModel.NintendoSdkMetaRootModel) : typeof(MetaModel.MetaRootModel);
                        System.Xml.Serialization.XmlSerializer serializer = new System.Xml.Serialization.XmlSerializer(modelType);
                        serializer.UnknownElement += serializer_UnknownElement;

                        tmpMs.Seek(0, SeekOrigin.Begin);

                        if (isNewMetaRootName)
                        {
                            var rootModel = (MetaModel.NintendoSdkMetaRootModel)serializer.Deserialize(tmpMs);
                            metaModel = rootModel.Core;
                        }
                        else
                        {
                            var rootModel = (MetaModel.MetaRootModel)serializer.Deserialize(tmpMs);
                            metaModel = rootModel.Core;
                        }
                    }
                }
                metaModel.MetaFileName = Path.GetFileNameWithoutExtension(param.InputMetaPath);
                metaModel.KernelVersion = param.KernelVersion;
                metaModel.NoCheckProgramId = param.NoCheckProgramId;

                npdmFile.Desc = descModel;
                npdmFile.Meta = metaModel;

                using (FileStream fs = new FileStream(param.OutputNpdmPath, FileMode.Create, FileAccess.Write))
                {
                    npdmFile.OutputNpdmFile(fs, param.DefaultCoreVersion);
                }
            }
            catch (Exception ex)
            {
                Environment.ExitCode = 1;
                Console.Error.WriteLine("Input Desc File: '{0}'", opt.Params.InputDescPath);
                Console.Error.WriteLine("Input Meta File: '{0}'", opt.Params.InputMetaPath);
                Console.Error.WriteLine(ex);
                if (System.IO.File.Exists(param.OutputNpdmPath))
                {
                    System.IO.File.Delete(param.OutputNpdmPath);
                }
                return;
            }
        }

        static void serializer_UnknownElement(object sender, XmlElementEventArgs e)
        {
            throw new ArgumentException(string.Format(Properties.Resources.Message_UnknownElement, e.Element.LocalName));
        }
    }
}
