﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeMeta
{
    public class NpdmFile
    {
        internal MetaModel.MetaModel Meta { get; set; }
        internal DescModel.OutputDescModel Desc { get; set; }

        private Aci AciData;
        private Meta MetaData;

        public NpdmFile()
        {
            AciData = new Aci();
        }

        public void OutputNpdmFile(FileStream fs, uint defaultCoreVersion)
        {
            if (Meta == null)
            {
                throw new ApplicationException(string.Format(Properties.Resources.Message_NotFoundFile, ".nmeta"));
            }
            else if (Desc == null)
            {
                throw new ApplicationException(string.Format(Properties.Resources.Message_NotFoundFile, ".desc"));
            }

            if (Meta.ProgramId == null)
            {
                throw new ApplicationException(string.Format(Properties.Resources.Message_NotFound, ".nmeta", "Core/ProgramId"));
            }

            MetaData = MetaGenerator.Merge(Meta, Desc, defaultCoreVersion);

            if (MetaData.SystemResourceSize != 0)
            {
                if (MetaData.ProcessAddressSpace != MetaModel.MetaModel.AddressSpace64Bit)
                {
                    throw new ApplicationException(string.Format(Properties.Resources.Message_OnlyAvailableFor64BitApplication, "Core/SystemResourceSize"));
                }

                if (Desc.KernelCapabilityDescriptor.MiscParams == null ||
                    (
                        Desc.KernelCapabilityDescriptor.MiscParams.ProgramTypeValue != DescModel.KcMiscParamsModel.ProgramType_Application &&
                        Desc.KernelCapabilityDescriptor.MiscParams.ProgramTypeValue != DescModel.KcMiscParamsModel.ProgramType_Applet
                    ))
                {
                    throw new ApplicationException(string.Format(Properties.Resources.Message_OnlyAvailableForApplicationsAndApplets, "Core/SystemResourceSize"));
                }

                if (Desc.KernelCapabilityDescriptor.EnableSystemCalls == null ||
                    ! Desc.KernelCapabilityDescriptor.EnableSystemCalls.Where(s => s.Name == "MapPhysicalMemory").Any() ||
                    ! Desc.KernelCapabilityDescriptor.EnableSystemCalls.Where(s => s.Name == "UnmapPhysicalMemory").Any())
                {
                    throw new ApplicationException(string.Format(Properties.Resources.Message_VirtualAddressMemoryFunctionsNotAvailable, "Core/SystemResourceSize"));
                }
            }

            AciData.ProgramId = Meta.ProgramIdValue;

            byte[] aciBinary = AciData.ExportBinary(Meta, Desc);
            byte[] acidBinary = Convert.FromBase64String(Desc.Acid);

            if (Meta.NoCheckProgramId == false)
            {
                if ((Meta.ProgramIdValue < Desc.ProgramIdMinValue) || (Meta.ProgramIdValue > Desc.ProgramIdMaxValue))
                {
                    throw new ApplicationException(string.Format(Properties.Resources.Message_OutOfRange, "Core/ProgramId, Core/ApplicationId", Meta.ProgramId));
                }
            }

            MetaData.AciSize = (uint)aciBinary.Length;
            MetaData.AcidSize = (uint)acidBinary.Length;

            BinaryWriter bw = new BinaryWriter(fs);
            bw.Write(MetaData.ExportBinary());
            bw.Write(acidBinary);
            uint alignSize = 0x10;
            uint diff = RoundUp(MetaData.AcidSize, alignSize) - MetaData.AcidSize;
            for (var i = 0; i < diff; i++)
            {
                bw.Write((byte)0);
            }
            bw.Write(aciBinary);
        }

        public uint RoundUp(uint target, uint align)
        {
            return ((target + (align - 1)) & ~(align - 1));
        }
    }
}
