﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Xml;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices;

namespace MakeMeta
{
    [StructLayout(LayoutKind.Sequential)]
    internal struct AciHeader
    {
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x4)]
        public byte[] Signature;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0xC)]
        public byte[] Reserved1;
        public ulong ProgramId;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x8)]
        public byte[] Reserved2;
        public uint FacOffset;
        public uint FacSize;
        public uint SacOffset;
        public uint SacSize;
        public uint KcOffset;
        public uint KcSize;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x8)]
        public byte[] Reserved3;
    }

    internal class Aci
    {
        private AciHeader Header;
        private const uint BinaryAlignSize = 0x10;

        private bool IsSetProgramId;
        public ulong ProgramId
        {
            get
            {
                return Header.ProgramId;
            }
            set
            {
                IsSetProgramId = true;
                Header.ProgramId = value;
            }
        }

        public DescModel.OutputDescModel DescModel { get; set; }
        public MetaModel.MetaModel MetaModel { get; set; }

        public Aci()
        {
            Header = new AciHeader();
            Header.Signature = System.Text.Encoding.ASCII.GetBytes("ACI0");
            Header.Reserved1 = new byte[0xC];
            Header.Reserved2 = new byte[0x8];
            Header.Reserved3 = new byte[0x8];
            IsSetProgramId = false;
        }

        public byte[] ExportBinary(MetaModel.MetaModel meta, DescModel.OutputDescModel desc)
        {
            if (!IsSetProgramId)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_NotFound, ".nmeta", "ProgramId"));
            }

            var fac = FacDataGenerator.Merge(meta, desc);
            var sac = SacDataGenerator.Merge(meta, desc);
            var kc = KcDataGenerator.Merge(meta, desc);

            KcDescriptor kcd = new KcDescriptor(desc);
            kcd.CheckCapabilities(kc);

            SacDescriptor sad = new SacDescriptor(desc);
            sad.CheckCapabilities(sac);

            uint HeaderSize = (uint)Marshal.SizeOf(typeof(AciHeader));

            byte[] facBinary = (fac != null) ? fac.ExportBinary() : null;
            byte[] sacBinary = (sac != null) ? sac.ExportBinary() : null;
            byte[] kcBinary = (kc != null) ? kc.ExportBinary() : null;

            Header.FacOffset = HeaderSize;
            Header.FacSize = (uint)((facBinary != null) ? facBinary.Length : 0);

            Header.SacOffset = Header.FacOffset + RoundUp(Header.FacSize, BinaryAlignSize);
            Header.SacSize = (uint)((sacBinary != null) ? sacBinary.Length : 0);

            Header.KcOffset = Header.SacOffset + RoundUp(Header.SacSize, BinaryAlignSize);
            Header.KcSize = (uint)((kcBinary != null) ? kcBinary.Length : 0);

            uint facBinSize = (Header.SacSize > 0 || Header.KcSize > 0) ? RoundUp(Header.FacSize, BinaryAlignSize) : Header.FacSize;
            uint sacBinSize = (Header.KcSize > 0) ? RoundUp(Header.SacSize, BinaryAlignSize) : Header.SacSize;
            uint kcBinSize = Header.KcSize;

            uint binarySize = HeaderSize + facBinSize + sacBinSize + kcBinSize;
            byte[] outputBinary = new byte[binarySize];

            GCHandle gch = GCHandle.Alloc(outputBinary, GCHandleType.Pinned);
            Marshal.StructureToPtr(Header, gch.AddrOfPinnedObject(), false);
            gch.Free();

            if (Header.FacSize > 0)
            {
                Array.Copy(facBinary, 0, outputBinary, Header.FacOffset, Header.FacSize);
            }

            if (Header.SacSize > 0)
            {
                Array.Copy(sacBinary, 0, outputBinary, Header.SacOffset, Header.SacSize);
            }

            if (Header.KcSize > 0)
            {
                Array.Copy(kcBinary, 0, outputBinary, Header.KcOffset, Header.KcSize);
            }

            return outputBinary;
        }

        public uint RoundUp(uint target, uint align)
        {
            return ((target + (align - 1)) & ~(align - 1));
        }
    }
}
