﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Runtime.CompilerServices;

[assembly: InternalsVisibleTo("MakeInitialProgramTest")]
namespace MakeInitialProgram
{
    using System;
    using Elf;

    /// <summary>
    /// エントリーポイントを定義します。
    /// </summary>
    internal static class Program
    {
        /// <summary>
        /// コンソールのエンコーディングを管理します。
        /// </summary>
        private class EncodingManager
        {
            private System.Text.Encoding OriginalEncoding { get; set; }

            internal EncodingManager()
            {
                OriginalEncoding = Console.OutputEncoding;
            }

            internal void ChangeOutputEncoding(System.Text.Encoding encoding)
            {
                Console.OutputEncoding = encoding;
            }

            internal void ResetOutputEncoding()
            {
                Console.OutputEncoding = OriginalEncoding;
            }

            ~EncodingManager()
            {
                Console.OutputEncoding = OriginalEncoding;
            }
        }

        private static EncodingManager encodingManager;

        /// <summary>
        /// エントリーポイントです。
        /// </summary>
        /// <param name="args">コマンドラインオプションです。</param>
        internal static void Main(string[] args)
        {
            System.Globalization.CultureInfo cultureInfo = System.Threading.Thread.CurrentThread.CurrentUICulture;
            if (cultureInfo.EnglishName.IndexOf("Japanese") < 0)
            {
                if (System.IO.Directory.Exists(System.IO.Path.Combine(System.IO.Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location), "en")))
                {
                    System.Threading.Thread.CurrentThread.CurrentUICulture = new System.Globalization.CultureInfo("en", false);
                }
            }

            encodingManager = new EncodingManager();

            try
            {
                CommandLineOption option = new CommandLineOption();

                option.Parse(args);

                if (option.Help)
                {
                    Console.Error.WriteLine(CommandLineOption.GetHelpMessage());

                    Environment.Exit(0);
                }

                if (option.IsUtf8)
                {
                    encodingManager.ChangeOutputEncoding(System.Text.Encoding.UTF8);
                }

                SystemCallInfo syscall = new SystemCallInfo();
                foreach (var filename in option.SvcHeader)
                {
                    syscall.Load(filename);
                }

                Logging.IsEnable = option.Verbose;

                var parameterInfo = new ParameterInfo(option.ParamFile, syscall, option.IsTestMode);

                Logging.WriteLine(parameterInfo);

                Logging.WriteLine();

                var elfInfo = new ElfInfo(option.Elf);

                Logging.WriteLine(elfInfo);

                Logging.WriteLine();

                Logging.WriteLine(Properties.Resources.LogMessage_Writing);

                Logging.WriteLine();

                var initalProgram = new InitialProgram(elfInfo, parameterInfo);

                initalProgram.Create(option.Output);

                Logging.WriteLine(initalProgram);

                Logging.WriteLine();

                Logging.WriteLine(Properties.Resources.LogMessage_OutputFile, option.Output);
            }
            catch (Exception ex)
            {
                Console.Error.WriteLine(Properties.Resources.Message_Error, ex.Message);

                Environment.Exit(1);
            }
        }
    }
}
