﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace MakeInitialProgram
{
    using System;
    using System.Collections.Generic;
    using System.Linq;

    /// <summary>
    /// 初期プログラム向けのバイナリデータを圧縮および圧縮解除するためのメソッドを提供します。
    /// </summary>
    internal static class LzCompression
    {
        private const int FooterLength = 12;

        private const int PositionOffset = 3;

        private const int JumpLengthBitOffset = 12;

        private const int MatchLengthBitOffset = 4;

        private const int JumpLengthBitMask = (1 << JumpLengthBitOffset) - 1;

        private const int MatchLengthBitMask = (1 << MatchLengthBitOffset) - 1;

        private const int MaxJumpLength = JumpLengthBitMask + PositionOffset;

        private const int MinMatchLength = PositionOffset;

        private const int MaxMatchLength = MatchLengthBitMask + PositionOffset;

        /// <summary>
        /// 初期プログラム向けのバイナリデータを圧縮します。
        /// </summary>
        /// <param name="data">バイナリデータです。</param>
        /// <returns>圧縮されたバイナリデータです。</returns>
        internal static byte[] Compress(byte[] data)
        {
            var compressedData = CompressCore(data);

            var tuple = UncompressCore(compressedData, (uint)data.Length);

            var uncompressedData = tuple.Item1;

            var compressedDataIndex = tuple.Item2;

            uint rawDataSize = (uint)data.Length - (uint)uncompressedData.Length;

            uint compressedDataSize = (uint)compressedData.Length - compressedDataIndex;

            uint alignedSize = GetAlignedSize(rawDataSize + compressedDataSize);

            var footer = PackFooter((uint)data.Length, rawDataSize, compressedDataSize);

            var ret = new byte[alignedSize + footer.Length];

            Array.Copy(data, ret, rawDataSize);

            Array.Copy(compressedData, compressedDataIndex, ret, rawDataSize, compressedDataSize);

            for (var i = rawDataSize + compressedDataSize; i < alignedSize; ++i)
            {
                ret[i] = 0xff;
            }

            Array.Copy(footer, 0, ret, alignedSize, footer.Length);

            return ret;
        }

        /// <summary>
        /// 初期プログラム向けに圧縮されたバイナリデータを圧縮解除します。
        /// </summary>
        /// <param name="data">圧縮されたバイナリデータです。</param>
        /// <returns>圧縮解除されたバイナリデータです。</returns>
        internal static byte[] Uncompress(byte[] data)
        {
            var footer = new byte[FooterLength];

            if (data.Length < footer.Length)
            {
                throw new ArgumentException(Properties.Resources.Message_InvalidCompressedData);
            }

            Array.Copy(data, data.Length - footer.Length, footer, 0, footer.Length);

            var originalDataSize = GetOriginalDataSizeFromFooter(footer, (uint)data.Length);

            var rawDataSize = GetRawDataSizeFromFooter(footer, (uint)data.Length);

            var compressedDataSize = GetCompressedDataSizeFromFooter(footer);

            if (data.Length < GetAlignedSize(rawDataSize + compressedDataSize) + footer.Length)
            {
                throw new ArgumentException(Properties.Resources.Message_InvalidFooter);
            }

            var compressedData = new byte[compressedDataSize];

            Array.Copy(data, rawDataSize, compressedData, 0, compressedData.Length);

            var uncompressedData = UncompressCore(compressedData, originalDataSize).Item1;

            if (originalDataSize != rawDataSize + uncompressedData.Length)
            {
                throw new ArgumentException(Properties.Resources.Message_DataIsBroken);
            }

            var originalData = new byte[originalDataSize];

            Array.Copy(data, originalData, rawDataSize);

            Array.Copy(uncompressedData, 0, originalData, rawDataSize, uncompressedData.Length);

            return originalData;
        }

        private static uint GetAlignedSize(uint value)
        {
            value += 3;

            return value - (value % 4);
        }

        private static byte[] PackAnchor(int jumpLength, int matchLength)
        {
            var anchor = (ushort)(((jumpLength - PositionOffset) & JumpLengthBitMask) | (((matchLength - PositionOffset) & MatchLengthBitMask) << JumpLengthBitOffset));

            return new byte[]
            {
                (byte)((anchor >> 0) & 0xff),
                (byte)((anchor >> 8) & 0xff)
            };
        }

        private static int GetJumpLengthFromAnchor(byte[] data)
        {
            return (UnpackAnchor(data) & JumpLengthBitMask) + PositionOffset;
        }

        private static int GetMatchLengthFromAnchor(byte[] data)
        {
            return ((UnpackAnchor(data) >> JumpLengthBitOffset) & MatchLengthBitMask) + PositionOffset;
        }

        private static ushort UnpackAnchor(byte[] data)
        {
            return (ushort)((data[1] << 8) | data[0]);
        }

        private static Tuple<int, int> GetLongestMatch(byte[] bytes, int position, List<int> candidates)
        {
            int tmpJumpLength = -1;

            int tmpMatchLength = 0;

            for (var i = candidates.Count - 1; i >= 0; --i)
            {
                var candidate = candidates[i];

                var length = 0;

                for (length = MinMatchLength; length < MaxMatchLength; ++length)
                {
                    if (candidate + length >= position)
                    {
                        break;
                    }

                    if (position + length >= bytes.Length)
                    {
                        break;
                    }

                    if (bytes[candidate + length] != bytes[position + length])
                    {
                        break;
                    }
                }

                if (length == MaxMatchLength)
                {
                    return Tuple.Create(position - candidate, length);
                }

                if (length > tmpMatchLength)
                {
                    tmpJumpLength = position - candidate;

                    tmpMatchLength = length;
                }
            }

            if (tmpMatchLength == 0)
            {
                throw new Exception(Properties.Resources.Message_DictionaryIsBroken);
            }

            return Tuple.Create(tmpJumpLength, tmpMatchLength);
        }

        private static byte[] PackFooter(uint originalDataSize, uint rawDataSize, uint compressedDataSize)
        {
            uint totalSize = GetAlignedSize(rawDataSize + compressedDataSize) + FooterLength;

            uint nonRawDataSize = totalSize - rawDataSize;

            byte nonDataSize = (byte)(nonRawDataSize - compressedDataSize);

            uint reducedSize = originalDataSize - totalSize;

            return new byte[]
            {
                (byte)((nonRawDataSize >> 0) & 0xff),
                (byte)((nonRawDataSize >> 8) & 0xff),
                (byte)((nonRawDataSize >> 16) & 0xff),
                (byte)((nonRawDataSize >> 24) & 0xff),
                (byte)((nonDataSize >> 0) & 0xff),
                (byte)((nonDataSize >> 8) & 0xff),
                (byte)((nonDataSize >> 16) & 0xff),
                (byte)((nonDataSize >> 24) & 0xff),
                (byte)((reducedSize >> 0) & 0xff),
                (byte)((reducedSize >> 8) & 0xff),
                (byte)((reducedSize >> 16) & 0xff),
                (byte)((reducedSize >> 24) & 0xff)
            };
        }

        private static uint GetOriginalDataSizeFromFooter(byte[] data, uint totalSize)
        {
            return totalSize + (((uint)data[11] << 24) | ((uint)data[10] << 16) | ((uint)data[9] << 8) | (uint)data[8]);
        }

        private static uint GetRawDataSizeFromFooter(byte[] data, uint totalSize)
        {
            return totalSize - GetNonRawDataSizeFromFooter(data);
        }

        private static uint GetCompressedDataSizeFromFooter(byte[] data)
        {
            return GetNonRawDataSizeFromFooter(data) - (((uint)data[7] << 24) | ((uint)data[6] << 16) | ((uint)data[5] << 8) | (uint)data[4]);
        }

        private static uint GetNonRawDataSizeFromFooter(byte[] data)
        {
            return ((uint)data[3] << 24) | ((uint)data[2] << 16) | ((uint)data[1] << 8) | (uint)data[0];
        }

        private static byte[] CompressCore(byte[] data)
        {
            var src = new byte[data.Length];

            var dst = new List<byte>();

            Array.Copy(data, src, data.Length);

            Array.Reverse(src);

            var ld = new LzDictionary();

            var index = 0;

            while (index < src.Length)
            {
                byte flag = 0;

                dst.Add(flag);

                var flagIndex = dst.Count - 1;

                for (var i = 0; i < 8 && index < src.Length; ++i)
                {
                    var list = ld.Find(src, index);

                    if (list == null)
                    {
                        dst.Add(src[index]);

                        var position = index - (MinMatchLength - 1);

                        if (position >= 0)
                        {
                            ld.Add(src, position);
                        }

                        ++index;
                    }
                    else
                    {
                        var tuple = GetLongestMatch(src, index, list);

                        var jumpLength = tuple.Item1;

                        var matchLength = tuple.Item2;

                        var packedAnchor = PackAnchor(jumpLength, matchLength);

                        dst.Add(packedAnchor[1]);

                        dst.Add(packedAnchor[0]);

                        for (var j = 0; j < matchLength; ++j)
                        {
                            ld.Add(src, index - (MinMatchLength - 1) + j);
                        }

                        index += matchLength;

                        flag |= (byte)(1 << (7 - i));
                    }
                }

                dst[flagIndex] = flag;
            }

            dst.Reverse();

            return dst.ToArray();
        }

        private static Tuple<byte[], uint> UncompressCore(byte[] data, uint originalDataSize)
        {
            var dst = new List<byte>();

            var index = (uint)data.Length;

            while (index > 0)
            {
                byte flag = data[--index];

                for (var i = 0; i < 8; ++i)
                {
                    if (index == 0 && i > 0)
                    {
                        break;
                    }

                    if ((flag & 0x80) == 0)
                    {
                        if (index < 1)
                        {
                            throw new ArgumentException(Properties.Resources.Message_DataIsBroken);
                        }

                        dst.Add(data[--index]);
                    }
                    else
                    {
                        if (index < 2)
                        {
                            throw new ArgumentException(Properties.Resources.Message_DataIsBroken);
                        }

                        index -= 2;

                        var packedAnchor = new byte[] { data[index], data[index + 1] };

                        var jumpLength = GetJumpLengthFromAnchor(packedAnchor);

                        var matchLength = GetMatchLengthFromAnchor(packedAnchor);

                        if (dst.Count < jumpLength || jumpLength < matchLength)
                        {
                            throw new ArgumentException(Properties.Resources.Message_DataIsBroken);
                        }

                        dst.AddRange(dst.GetRange(dst.Count - jumpLength, matchLength));

                        if ((originalDataSize - dst.Count) < index)
                        {
                            dst.Reverse();

                            return Tuple.Create(dst.ToArray(), index);
                        }
                    }

                    flag <<= 1;
                }
            }

            dst.Reverse();

            return Tuple.Create(dst.ToArray(), 0u);
        }

        private sealed class LzDictionary
        {
            internal LzDictionary()
            {
                this.Dictionary = new Dictionary<int, List<int>>();
            }

            private Dictionary<int, List<int>> Dictionary { get; set; }

            internal List<int> Find(byte[] bytes, int position)
            {
                if (position + MinMatchLength > bytes.Length)
                {
                    return null;
                }

                var key = ConvertBytesToKey(bytes, position);

                if (!this.Dictionary.ContainsKey(key))
                {
                    return null;
                }

                var list = this.Dictionary[key].FindAll(x => x >= position - MaxJumpLength);

                if (list.Count == 0)
                {
                    return null;
                }

                return list;
            }

            internal LzDictionary Add(byte[] bytes, int position)
            {
                var key = ConvertBytesToKey(bytes, position);

                if (!this.Dictionary.ContainsKey(key))
                {
                    this.Dictionary.Add(key, new List<int>());
                }

                this.Dictionary[key].RemoveAll(x => x < position - MaxJumpLength);
                this.Dictionary[key].Add(position);

                return this;
            }

            private static int ConvertBytesToKey(byte[] bytes, int position)
            {
                var tmp = new byte[sizeof(int)];

                Array.Copy(bytes, position, tmp, 0, MinMatchLength);

                return BitConverter.ToInt32(tmp, 0);
            }
        }
    }
}
