﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace MakeInitialProgram
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using Elf;

    /// <summary>
    /// 初期プログラムを表します。
    /// </summary>
    internal sealed class InitialProgram
    {
        internal const int MaxNameLength = 11;

        /// <summary>
        /// InitialProgram クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="elfInfo">初期プログラムの作成元となるELFデータです。</param>
        /// <param name="parameterInfo">初期プログラムの作成時に使用するパラメータです。</param>
        internal InitialProgram(ElfInfo elfInfo, ParameterInfo parameterInfo)
        {
            this.ExSegmentInfoReader = new BinarySegmentInfoReader("EX Binary", elfInfo.GetExElfSegmentInfo());

            this.RoSegmentInfoReader = new BinarySegmentInfoReader("RO Binary", elfInfo.GetRoElfSegmentInfo());

            this.RwSegmentInfoReader = new BinarySegmentInfoReader("RW Binary", elfInfo.GetRwElfSegmentInfo());

            this.ZiSegmentInfoReader = new RegionSegmentInfoReader("ZI Region", elfInfo.GetZiElfSegmentInfo());

            this.Flag = (byte)((this.ExSegmentInfoReader.IsCompressed() ? 0x01 : 0x00) |
                               (this.RoSegmentInfoReader.IsCompressed() ? 0x02 : 0x00) |
                               (this.RwSegmentInfoReader.IsCompressed() ? 0x04 : 0x00) |
                               (elfInfo.Machine.Is64bit() ? 0x08 : 0x00) |
                               (elfInfo.Architecture == ElfArchitectureType.Elf64 ? 0x10 : 0x00));

            this.Name = CalculateInitialProgramName(elfInfo, parameterInfo);

            this.ProgramId = parameterInfo.ProgramId;

            this.Version = parameterInfo.Version;

            this.StackSize = parameterInfo.StackSize;

            this.Priority = parameterInfo.Priority;

            this.IdealProcessor = parameterInfo.IdealProcessor;

            if (parameterInfo.UseSecureMemory)
            {
                this.Flag |= (1 << 5);
            }

            this.Capability = parameterInfo.Capability;

            // セクションのアラインチェック
            if (elfInfo.FileType == ElfFileType.SharedObjectFile)
            {
                foreach (var section in elfInfo.SectionInfos)
                {
                    if (section.AddressAlign > 0x1000)
                    {
                        throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidSectionAlign, section.SectionName, section.AddressAlign));
                    }
                }
            }

            // セグメントのアラインチェック
            if (elfInfo.GetExElfSegmentInfo().VirtualAddress != 0)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidSegmentAddress0, "EX", elfInfo.GetExElfSegmentInfo().VirtualAddress));
            }
            if ((elfInfo.GetRoElfSegmentInfo().VirtualAddress & 0xFFF) > 0)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidSegmentAlign, "RO", "4KB", elfInfo.GetRoElfSegmentInfo().VirtualAddress));
            }
            if ((elfInfo.GetRwElfSegmentInfo().VirtualAddress & 0xFFF) > 0)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidSegmentAlign, "RW", "4KB", elfInfo.GetRwElfSegmentInfo().VirtualAddress));
            }
            if ((elfInfo.GetZiElfSegmentInfo().VirtualAddress & 0xF) > 0)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidSegmentAlign, "ZI", "16 Byte", elfInfo.GetZiElfSegmentInfo().VirtualAddress));
            }
        }

        private BinarySegmentInfoReader ExSegmentInfoReader { get; set; }

        private BinarySegmentInfoReader RoSegmentInfoReader { get; set; }

        private BinarySegmentInfoReader RwSegmentInfoReader { get; set; }

        private RegionSegmentInfoReader ZiSegmentInfoReader { get; set; }

        private byte Flag { get; set; }

        private string Name { get; set; }

        private ulong ProgramId { get; set; }

        private uint Version { get; set; }

        private uint StackSize { get; set; }

        private byte Priority { get; set; }

        private byte IdealProcessor { get; set; }

        private uint[] Capability { get; set; }

        /// <summary>
        /// 現在のオブジェクトを表す文字列を返します。
        /// </summary>
        /// <returns>
        /// 現在のオブジェクトを表す文字列です。
        /// </returns>
        public override string ToString()
        {
            var sb = new StringBuilder();

            sb.Append("Initial Program Information:");

            foreach (var reader in new[] { this.ExSegmentInfoReader, this.RoSegmentInfoReader, this.RwSegmentInfoReader, this.ZiSegmentInfoReader })
            {
                if (reader.Exists())
                {
                    sb.AppendLine();

                    sb.Append(reader.ToString());
                }
            }

            return sb.ToString();
        }

        /// <summary>
        /// 初期プログラムをファイルに書き出します。
        /// </summary>
        /// <param name="fileName">初期プログラムの書き出しとなるファイルのパスです。</param>
        internal void Create(string fileName)
        {
            using (var fs = File.Create(fileName))
            using (var bw = new BinaryWriter(fs))
            {
                bw.Write(new byte[] { (byte)'K', (byte)'I', (byte)'P', (byte)'1' });

                for (var i = 0; i < this.Name.Length; ++i)
                {
                    bw.Write((byte)this.Name[i]);
                }

                for (var i = this.Name.Length; i < MaxNameLength; ++i)
                {
                    bw.Write((byte)0);
                }

                // 12 文字目は必ず ヌル文字
                bw.Write((byte)0);

                bw.Write(this.ProgramId);

                bw.Write(this.Version);

                bw.Write(this.Priority);

                bw.Write(this.IdealProcessor);

                bw.Write((byte)0); // reserved

                bw.Write(this.Flag);

                bw.Write(this.ExSegmentInfoReader.VirtualAddress);

                bw.Write(this.ExSegmentInfoReader.MemorySize);

                bw.Write(this.ExSegmentInfoReader.Binary.Count());

                bw.Write(0u); // AffinityMask が削除された分

                bw.Write(this.RoSegmentInfoReader.VirtualAddress);

                bw.Write(this.RoSegmentInfoReader.MemorySize);

                bw.Write(this.RoSegmentInfoReader.Binary.Count());

                bw.Write(this.StackSize);

                bw.Write(this.RwSegmentInfoReader.VirtualAddress);

                bw.Write(this.RwSegmentInfoReader.MemorySize);

                bw.Write(this.RwSegmentInfoReader.Binary.Count());

                bw.Write(0u); // reserved

                bw.Write(this.ZiSegmentInfoReader.VirtualAddress);

                bw.Write(this.ZiSegmentInfoReader.MemorySize);

                for (var i = 0; i < 10; ++i)
                {
                    bw.Write(0u); // reserved
                }

                foreach (var flag in this.Capability)
                {
                    bw.Write(flag);
                }

                foreach (var reader in new[] { this.ExSegmentInfoReader, this.RoSegmentInfoReader, this.RwSegmentInfoReader })
                {
                    if (reader.Exists())
                    {
                        bw.Write(reader.Binary.ToArray());
                    }
                }
            }
        }

        private static string CalculateInitialProgramName(ElfInfo elfInfo, ParameterInfo parameterInfo)
        {
            var name = parameterInfo.Name != null ? parameterInfo.Name : Path.GetFileName(elfInfo.FileName).Split('.')[0];

            if (name.Length > MaxNameLength)
            {
                name = name.Substring(0, MaxNameLength);
            }

            return name;
        }

        private class RegionSegmentInfoReader
        {
            internal RegionSegmentInfoReader(string name, ElfSegmentInfo info)
            {
                this.Name = name;

                if (info == null)
                {
                    this.ExistenceFlag = false;

                    this.VirtualAddress = 0;

                    this.MemorySize = 0;

                    this.SegmentNames = (new List<string>()).AsReadOnly();
                }
                else
                {
                    CheckSanityOf(this.Name, info);

                    this.ExistenceFlag = true;

                    this.VirtualAddress = (uint)info.VirtualAddress;

                    this.MemorySize = (uint)info.MemorySize;

                    this.SegmentNames = info.SegmentNames;
                }
            }

            internal string Name { get; private set; }

            internal uint VirtualAddress { get; private set; }

            internal uint MemorySize { get; private set; }

            internal IReadOnlyList<string> SegmentNames { get; private set; }

            private bool ExistenceFlag { get; set; }

            /// <summary>
            /// 現在のオブジェクトを表す文字列を返します。
            /// </summary>
            /// <returns>
            /// 現在のオブジェクトを表す文字列です。
            /// </returns>
            public override string ToString()
            {
                var sb = new StringBuilder();

                sb.AppendLine(string.Format("  {0}:", this.Name));

                sb.AppendLine(string.Format("    Contains:     {0}", string.Join(",", this.SegmentNames)));

                sb.AppendLine(string.Format("    Address:      0x{0}", this.VirtualAddress.ToString("x8")));

                sb.AppendFormat("    MemorySize:   0x{0}", this.MemorySize.ToString("x8"));

                return sb.ToString();
            }

            internal bool Exists()
            {
                return this.ExistenceFlag;
            }

            private static void CheckSanityOf(string name, ElfSegmentInfo info)
            {
                if (info == null)
                {
                    return;
                }

                if (info.SegmentSize > uint.MaxValue)
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidSegmentSize, name));
                }

                if (info.VirtualAddress > uint.MaxValue)
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidVirtualAddress, name));
                }

                if (info.MemorySize > uint.MaxValue)
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidMemorySize, name));
                }
            }
        }

        private sealed class BinarySegmentInfoReader : RegionSegmentInfoReader
        {
            internal BinarySegmentInfoReader(string name, ElfSegmentInfo info) : base(name, info)
            {
                if (info == null)
                {
                    this.SegmentSize = 0;

                    this.CompressionFlag = false;

                    this.Binary = (new List<byte>()).AsReadOnly();
                }
                else
                {
                    this.SegmentSize = (uint)info.SegmentSize;

                    var originalBinary = info.GetContents();

                    var compressedBinary = CompressBinary(this.Name, originalBinary);

                    if (compressedBinary.Length < originalBinary.Length)
                    {
                        this.CompressionFlag = true;

                        this.Binary = Array.AsReadOnly(compressedBinary);
                    }
                    else
                    {
                        this.CompressionFlag = false;

                        this.Binary = Array.AsReadOnly(originalBinary);
                    }
                }
            }

            internal IReadOnlyList<byte> Binary { get; private set; }

            private uint SegmentSize { get; set; }

            private bool CompressionFlag { get; set; }

            /// <summary>
            /// 現在のオブジェクトを表す文字列を返します。
            /// </summary>
            /// <returns>
            /// 現在のオブジェクトを表す文字列です。
            /// </returns>
            public override string ToString()
            {
                var sb = new StringBuilder();

                sb.AppendLine(string.Format("  {0}:", this.Name));

                sb.AppendLine(string.Format("    Contains:     {0}", string.Join(",", this.SegmentNames)));

                sb.AppendLine(string.Format("    Address:      0x{0}", this.VirtualAddress.ToString("x8")));

                sb.AppendLine(string.Format("    MemorySize:   0x{0}", this.MemorySize.ToString("x8")));

                if (this.IsCompressed())
                {
                    var ratio = this.Binary.Count() / (double)this.SegmentSize * 100;

                    sb.AppendLine(string.Format("    BinarySize:   0x{0} ({1,5:F1}%)", this.Binary.Count().ToString("x8"), ratio));
                }
                else
                {
                    sb.AppendLine(string.Format("    BinarySize:   0x{0}", this.Binary.Count().ToString("x8")));
                }

                sb.AppendFormat("    IsCompressed: {0}", this.IsCompressed());

                return sb.ToString();
            }

            internal bool IsCompressed()
            {
                return this.CompressionFlag;
            }

            private static byte[] CompressBinary(string name, byte[] data)
            {
                var compressedData = LzCompression.Compress(data);

                var uncompressedData = LzCompression.Uncompress(compressedData);

                if (!uncompressedData.SequenceEqual(data))
                {
                    throw new Exception(string.Format(Properties.Resources.Message_MissCompress, name));
                }

                return compressedData;
            }
        }
    }
}
