﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace MakeInitialProgram.Elf
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.IO;
    using System.Linq;
    using System.Text;

    /// <summary>
    /// ELF ファイルのセクションを表します。
    /// </summary>
    public sealed class ElfSectionInfo
    {
        private ElfSectionInfo(ElfInfo elfInfo, long offset)
        {
            this.FileName = elfInfo.FileName;

            this.Architecture = elfInfo.Architecture;

            this.ByteOrder = elfInfo.ByteOrder;

            using (var fs = File.OpenRead(elfInfo.FileName))
            using (var er = new ElfBinaryReader(fs, elfInfo.Architecture, elfInfo.ByteOrder))
            {
                fs.Seek(offset, SeekOrigin.Begin);

                this.SectionNameOffset = er.ReadUInt32();

                this.SectionType = ConvertToElfSectionType(er.ReadUInt32());

                er.ReadWord(); // section flags

                this.VirtualAddress = er.ReadWord();

                this.FileOffset = er.ReadWord();

                this.SectionSize = er.ReadWord();

                er.ReadUInt32(); // link

                er.ReadUInt32(); // Miscellaneous information

                this.AddressAlign = er.ReadWord();
            }

            this.SectionName = string.Empty;
        }

        /// <summary>
        /// ELF ファイルのパスを取得します。
        /// </summary>
        public string FileName { get; private set; }

        /// <summary>
        /// アーキテクチャタイプを取得します。
        /// </summary>
        public ElfArchitectureType Architecture { get; private set; }

        /// <summary>
        /// バイトオーダタイプを取得します。
        /// </summary>
        public ElfByteOrderType ByteOrder { get; private set; }

        /// <summary>
        /// セクション名の位置を示すオフセットを取得します。
        /// </summary>
        public uint SectionNameOffset { get; private set; }

        /// <summary>
        /// セクションタイプを取得します。
        /// </summary>
        public ElfSectionType SectionType { get; private set; }

        /// <summary>
        /// バーチャルアドレスを取得します。
        /// </summary>
        public ulong VirtualAddress { get; private set; }

        /// <summary>
        /// ファイルオフセットを取得します。
        /// </summary>
        public ulong FileOffset { get; private set; }

        /// <summary>
        /// セクションサイズを取得します。
        /// </summary>
        public ulong SectionSize { get; private set; }

        /// <summary>
        /// セクション名を取得します。
        /// </summary>
        public string SectionName { get; private set; }

        public ulong AddressAlign { get; private set; }

        /// <summary>
        /// セクションからバイナリデータを読み取ります。
        /// </summary>
        /// <returns>読み取ったバイナリデータです。</returns>
        public byte[] GetContents()
        {
            if (this.FileOffset > long.MaxValue)
            {
                throw new ArgumentException(Properties.Resources.Message_InvalidSectionFileOffset);
            }

            if (this.SectionSize > int.MaxValue)
            {
                throw new ArgumentException(Properties.Resources.Message_InvalidSectionSize);
            }

            using (var fs = File.OpenRead(this.FileName))
            using (var br = new BinaryReader(fs))
            {
                fs.Seek((long)this.FileOffset, SeekOrigin.Begin);

                return br.ReadBytes((int)this.SectionSize);
            }
        }

        /// <summary>
        /// ELF データのセクションヘッダテーブルからセクションを読み取ります。
        /// </summary>
        /// <param name="elfInfo">ELF データです。</param>
        /// <param name="offset">セクションヘッダテーブルの位置を示すオフセットです。</param>
        /// <param name="size">セクションヘッダテーブルのエントリサイズです。</param>
        /// <param name="length">セクションヘッダテーブルのエントリ数です。</param>
        /// <param name="sectionInfos">ストリングテーブルの位置を示すインデックスです。</param>
        /// <returns>読み取ったセクションです。</returns>
        internal static IReadOnlyList<ElfSectionInfo> GetElfSectionInfos(ElfInfo elfInfo, ulong offset, ushort size, ushort length, ushort stringTableIndex)
        {
            if (!IsValidElfSectionHeaderEntrySize(size, elfInfo.Architecture))
            {
                throw new ArgumentException(Properties.Resources.Message_InvalidSectionHeaderEntrySize);
            }

            if (offset > long.MaxValue)
            {
                throw new ArgumentException(Properties.Resources.Message_InvalidSectionHeaderTableOffset);
            }

            var list = new List<ElfSectionInfo>();

            for (long i = 0; i < length; ++i)
            {
                var entryOffset = (long)offset + (size * i);

                list.Add(new ElfSectionInfo(elfInfo, entryOffset));
            }

            if (stringTableIndex != 0)
            {
                if ((stringTableIndex >= list.Count) || (list[stringTableIndex].SectionType != ElfSectionType.StringTable))
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidStringTableIndex, stringTableIndex));
                }

                var bytes = list[stringTableIndex].GetContents();

                foreach (var si in list)
                {
                    si.SectionName = GetSectionNameFromStringTable(bytes, si.SectionNameOffset);
                }
            }

            return new ReadOnlyCollection<ElfSectionInfo>(list);
        }

        private static ElfSectionType ConvertToElfSectionType(uint value)
        {
            switch (value)
            {
                case 3: return ElfSectionType.StringTable;
                case 7: return ElfSectionType.Note;
                default: return ElfSectionType.Unknown;
            }
        }

        private static bool IsValidElfSectionHeaderEntrySize(ushort size, ElfArchitectureType architecture)
        {
            if (architecture == ElfArchitectureType.Elf32 && size == 40)
            {
                return true;
            }

            if (architecture == ElfArchitectureType.Elf64 && size == 64)
            {
                return true;
            }

            return false;
        }

        private static string GetSectionNameFromStringTable(byte[] bytes, uint offset)
        {
            var sb = new StringBuilder();

            for (uint i = offset; i < bytes.Length; ++i)
            {
                if (bytes[i] > 0)
                {
                    sb.Append((char)bytes[i]);
                }
                else
                {
                    return sb.ToString();
                }
            }

            throw new ArgumentException(Properties.Resources.Message_FailedToGetSectionName);
        }
    }
}
