﻿using System;
using System.Collections.Generic;
using System.Text;
using Nintendo.Foundation.IO;
using System.IO;
using CommandUtility;
using YamlDotNet.Serialization;
using System.Linq;

namespace MakeFirmwareArchive
{
    public class ExecuteListArguments
    {
        [CommandLineOption('o', "output", Description = " ", IsRequired = true)]
        public string OutputPath { get; set; }

        [CommandLineOption('i', "input", Description = " ", IsRequired = true)]
        public string InputPath { get; set; }

        [CommandLineOption("list", Description = " ", IsRequired = true)]
        public string ListPath { get; set; }
    }

    public class ExecuteCommandList
    {
        public List<ExecuteCommand> Commands { get; set; }
    }

    public class ExecuteCommand
    {
        public string Name { get; set; }
        public string PartitionName { get; set; }
        public string File { get; set; }
        public Dictionary<string, string> Parameters { get; set; }
    }

    class ExecuteListCommand
    {
        public static void AddFiles(ExecuteCommand command, FirmwareArchiveDirectory archiveDirectory)
        {
            if (!FirmwareArchivePartition.IsValidPartitionName(command.PartitionName))
            {
                throw new Exception(string.Format("Invalid partition name: {0}", command.PartitionName));
            }

            var partitionName = command.PartitionName;
            var file = command.File;

            AddFilesToFirmwareArchiveDirectory(archiveDirectory, partitionName, file);
        }

        private static void AddFilesToFirmwareArchiveDirectory(FirmwareArchiveDirectory archiveDirectory, string partitionName, string file)
        {
            DirectoryInfo directoryInfo = archiveDirectory.GetDirectoryInfo(partitionName);
            if (!directoryInfo.Exists)
            {
                directoryInfo.Create();
            }

            DirectoryInfo searchDirectory = new DirectoryInfo(Path.GetDirectoryName(file));

            if (!searchDirectory.Exists && file.EndsWith("*"))
            {
                Console.WriteLine("[WARNING] file not found: {0}", file);
                return;
            }

            var files = searchDirectory.EnumerateFiles(Path.GetFileName(file)).ToList();

            if (files.Count == 0 && file != "*")
            {
                throw new Exception(string.Format("File not found: {0}", file));
            }

            foreach (var f in files)
            {
                if (File.Exists(Path.Combine(directoryInfo.FullName, f.Name)))
                {
                    Console.WriteLine("replaced: {0}/{1}", partitionName, f.Name);
                }
                else
                {
                    Console.WriteLine("added: {0}/{1}", partitionName, f.Name);
                }

                f.CopyTo(Path.Combine(directoryInfo.FullName, f.Name), true);
            }
        }

        public static void RemoveFiles(ExecuteCommand command, FirmwareArchiveDirectory archiveDirectory)
        {
            DirectoryInfo directoryInfo = archiveDirectory.GetDirectoryInfo(command.PartitionName);
            if (!directoryInfo.Exists)
            {
                throw new Exception(string.Format("Directory not found: {0}", directoryInfo.Name));
            }

            var files = directoryInfo.EnumerateFiles(command.File).ToList();
            if (files.Count == 0 && command.File != "*")
            {
                throw new Exception(string.Format("File not found: {0}", command.File));
            }

            foreach (var file in files)
            {
                Console.WriteLine("deleted: {0}/{1}", command.PartitionName, file.Name);
                file.Delete();
            }
        }

        public static ExecuteCommandList parseCommand(FileInfo listPath)
        {
            using (var input = new StreamReader(listPath.FullName, Encoding.UTF8))
            {
                var yamlDeserializer = new Deserializer();
                var config = yamlDeserializer.Deserialize<ExecuteCommandList>(input);
                return config;
            }
        }

        public static void ExecuteList(FileInfo outputPath, FileInfo inputPath, FileInfo listPath)
        {
            using (var tempHolder = new TemporaryFileHolder("MakeFirmwareArhive"))
            {
                var extractedNfa = tempHolder.CreateTemporaryDirectory("ExtractedNfa");
                var archiveDirectory = FirmwareArchiveDirectory.FromFirmwareArchive(extractedNfa, inputPath);

                var commandList = parseCommand(listPath);

                foreach (var command in commandList.Commands)
                {
                    if (command.Name == "Add")
                    {
                        AddFiles(command, archiveDirectory);
                    }
                    else if (command.Name == "Remove")
                    {
                        RemoveFiles(command, archiveDirectory);
                    }
                    else if (command.Name == "ReplaceBootImagePackage")
                    {
                        ReplaceBootImagePackage(command, archiveDirectory);
                    }
                    else if (command.Name == "CopyNsp")
                    {
                        CopyNsp(command, archiveDirectory);
                    }
                    else
                    {
                        throw new Exception(string.Format("Invalid command name: {0}", command.Name));
                    }
                }

                archiveDirectory.ArchiveNfa(outputPath);
            }
        }

        private static void ReplaceBootImagePackage(ExecuteCommand command, FirmwareArchiveDirectory archiveDirectory)
        {
            DirectoryInfo directoryInfo = archiveDirectory.GetDirectoryInfo(command.PartitionName);
            if (!directoryInfo.Exists)
            {
                throw new Exception(string.Format("Directory not found: {0}", directoryInfo.Name));
            }

            var files = directoryInfo.EnumerateFiles(command.File).ToList();
            if (files.Count == 0)
            {
                throw new Exception(string.Format("File not found: {0}", command.File));
            }

            var file = files.Single();

            var parameters = command.Parameters;
            if (command.Parameters == null)
            {
                throw new Exception(string.Format("Parameter Not Found: Command={0}", command.Name));
            }

            if (!(parameters.ContainsKey("Bct") &&
                   parameters.ContainsKey("Package1") &&
                   parameters.ContainsKey("Package2")))
            {
                throw new Exception(string.Format("{0} command needs parameters: Bct, Package1, Package2", command.Name));
            }

            var bct = FileUtility.MakeExistedFileInfo("Bct File", parameters["Bct"]);
            var package1 = FileUtility.MakeExistedFileInfo("Package1 File", parameters["Package1"]);
            var package2 = FileUtility.MakeExistedFileInfo("Package2 File", parameters["Package2"]);
            var bctMariko = FileUtility.MakeExistedFileInfo("Bct File for Mariko", parameters["Bct-M"]);
            var package1Mariko = FileUtility.MakeExistedFileInfo("Package1 File for Mariko", parameters["Package1-M"]);

            Console.WriteLine("Replace BootImagePackage: {0}/{1}", command.PartitionName, file.Name);

            SdkTool.Execute("ReplaceBootImagePackage.exe",
                "-i", file.FullName,
                "--overwrite",
                "--bct", bct.FullName,
                "--package1", package1.FullName,
                "--package2", package2.FullName,
                "--bct-mariko", bctMariko.FullName,
                "--package1-mariko", package1Mariko.FullName);
        }

        private static void CopyNsp(ExecuteCommand command, FirmwareArchiveDirectory archiveDirectory)
        {
            using (var tempHolder = new TemporaryFileHolder("MakeFirmwareArhive"))
            {
                var extractedNfa = tempHolder.CreateTemporaryDirectory("ExtractedNfa");
                var inputPath = new FileInfo(command.File);
                var inputArchiveDirectory = FirmwareArchiveDirectory.FromFirmwareArchive(extractedNfa, inputPath);
                AddFilesToFirmwareArchiveDirectory(archiveDirectory, command.PartitionName, inputArchiveDirectory.SystemDirectory.ToString() + @"\\*");
            }
        }
    }
}
