﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using System.Xml.Serialization;

namespace MakeDesc
{
    internal class Program
    {
        private static void Main(string[] args)
        {
            System.Globalization.CultureInfo cultureInfo = System.Threading.Thread.CurrentThread.CurrentUICulture;
            if (cultureInfo.EnglishName.IndexOf("Japanese") < 0)
            {
                if (System.IO.Directory.Exists(System.IO.Path.Combine(System.IO.Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location), "en")))
                {
                    System.Threading.Thread.CurrentThread.CurrentUICulture = new System.Globalization.CultureInfo("en", false);
                }
            }

            MakeDescArgs opt = new MakeDescArgs();
            try
            {
                if (!opt.ParseArgs(args))
                {
                    return;
                }
            }
            catch
            {
                Environment.ExitCode = 1;
                return;
            }

            MakeDescParams param = opt.Params;

            try
            {
                SystemCallInfo sysCallInfo = null;

                if (param.SvcHeaderPath != null)
                {
                    sysCallInfo = new SystemCallInfo();
                    foreach (var path in param.SvcHeaderPath)
                    {
                        sysCallInfo.Load(path);
                    }
                }

                DescFile descFile = new DescFile(sysCallInfo, param.KernelVersion, param.DescVersion);

                DescModel.InputDescModel inputModel;
                using (FileStream fs = new FileStream(param.InputDescPath, FileMode.Open, FileAccess.Read))
                {
                    System.Xml.Serialization.XmlSerializer serializer = new System.Xml.Serialization.XmlSerializer(typeof(DescModel.InputDescModel));
                    serializer.UnknownElement += serializer_UnknownElement;

                    inputModel = (DescModel.InputDescModel)serializer.Deserialize(fs);
                }

                descFile.ImportDescFile(inputModel);

                KeyManager keyManager = new KeyManager();
                if (param.KeyPath != null && inputModel.NeedDescSignatureValue)
                {
                    keyManager.LoadPrivateKeyFile(param.KeyPath);
                }
                descFile.KeyManager = keyManager;

                using (MemoryStream outputDescFile = new MemoryStream())
                {
                    descFile.OutputDescFile(outputDescFile);
                    outputDescFile.Flush();
                    outputDescFile.Seek(0, SeekOrigin.Begin);

                    bool doSkip = false;

                    if (File.Exists(param.OutputDescPath))
                    {
                        using (FileStream dstFs = new FileStream(param.OutputDescPath, FileMode.Open, FileAccess.Read, FileShare.Read))
                        {
                            doSkip = CmpAcidBinary(outputDescFile, dstFs);

                            if (doSkip)
                            {
                                doSkip = CmpDescFile(outputDescFile, dstFs);
                            }
                        }
                    }

                    if (!doSkip)
                    {
                        using (FileStream fs = new FileStream(param.OutputDescPath, FileMode.Create, FileAccess.Write, FileShare.None))
                        {
                            outputDescFile.CopyTo(fs);
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Environment.ExitCode = 1;
                Console.Error.WriteLine("Input FileName: '{0}'", opt.Params.InputDescPath);
                Console.Error.WriteLine(ex);
                return;
            }
        }

        /// <summary>
        /// desc ファイルの Acid バイナリの署名に関係しない部分を比較します。
        /// </summary>
        /// <param name="src">出力予定の desc ファイル</param>
        /// <param name="dst">出力先の desc ファイル</param>
        /// <returns>バイナリが等しければ True、異なっていれば False</returns>
        static bool CmpAcidBinary(Stream src, Stream dst)
        {
            long srcCurrentOffset = src.Position;
            long dstCurrentOffset = dst.Position;

            System.Xml.Serialization.XmlSerializer srcSerializer = new System.Xml.Serialization.XmlSerializer(typeof(DescModel.OutputDescModel));
            System.Xml.Serialization.XmlSerializer dstSerializer = new System.Xml.Serialization.XmlSerializer(typeof(DescModel.OutputDescModel));
            srcSerializer.UnknownElement += serializer_UnknownElement;
            dstSerializer.UnknownElement += serializer_UnknownElement;

            DescModel.OutputDescModel srcModel = (DescModel.OutputDescModel)srcSerializer.Deserialize(src);

            DescModel.OutputDescModel dstModel;
            try
            {
                dstModel = (DescModel.OutputDescModel)dstSerializer.Deserialize(dst);
            }
            catch (Exception)
            {
                // 無効な desc なので、上書きして問題ない
                src.Seek(srcCurrentOffset, SeekOrigin.Begin);
                dst.Seek(dstCurrentOffset, SeekOrigin.Begin);
                return false;
            }

            src.Seek(srcCurrentOffset, SeekOrigin.Begin);
            dst.Seek(dstCurrentOffset, SeekOrigin.Begin);

            byte[] srcAcidBinary = Convert.FromBase64String(srcModel.Acid);
            byte[] dstAcidBinary = Convert.FromBase64String(dstModel.Acid);
            const int AcidContentsOffset = 0x200;
            if (srcAcidBinary.Length != dstAcidBinary.Length)
            {
                return false;
            }

            byte[] srcAcidData = new byte[srcAcidBinary.Length - AcidContentsOffset];
            byte[] dstAcidData = new byte[dstAcidBinary.Length - AcidContentsOffset];
            Array.Copy(srcAcidBinary, AcidContentsOffset, srcAcidData, 0, srcAcidBinary.Length - AcidContentsOffset);
            Array.Copy(dstAcidBinary, AcidContentsOffset, dstAcidData, 0, dstAcidBinary.Length - AcidContentsOffset);
            return srcAcidData.SequenceEqual(dstAcidData);
        }

        /// <summary>
        /// desc ファイルの Acid と RSAKeyValue の項目以外の部分を比較します。
        /// </summary>
        /// <param name="src">出力予定の desc ファイル</param>
        /// <param name="dst">出力先の desc ファイル</param>
        /// <returns>ファイル内の内容が等しければ True、異なっていれば False</returns>
        static bool CmpDescFile(Stream src, Stream dst)
        {
            StreamReader srcReader = new StreamReader(src);
            StreamReader dstReader = new StreamReader(dst);
            long srcCurrentOffset = src.Position;
            long dstCurrentOffset = dst.Position;
            bool srcHasKey = false;
            bool dstHasKey = false;
            while (!srcReader.EndOfStream && !dstReader.EndOfStream)
            {
                string srcLine = srcReader.ReadLine();
                string dstLine = dstReader.ReadLine();

                if (srcLine.IndexOf("<RSAKeyValue>") >= 0)
                {
                    srcHasKey = true;
                    while (!srcReader.EndOfStream)
                    {
                        srcLine = srcReader.ReadLine();
                        if (srcLine.IndexOf("</RSAKeyValue>") >= 0)
                        {
                            break;
                        }
                    }

                    if (srcReader.EndOfStream)
                    {
                        break;
                    }

                    srcLine = srcReader.ReadLine();
                }

                if (dstLine.IndexOf("<RSAKeyValue>") >= 0)
                {
                    dstHasKey = true;
                    while (!dstReader.EndOfStream)
                    {
                        dstLine = dstReader.ReadLine();
                        if (dstLine.IndexOf("</RSAKeyValue>") >= 0)
                        {
                            break;
                        }
                    }

                    if (dstReader.EndOfStream)
                    {
                        break;
                    }

                    dstLine = dstReader.ReadLine();
                }

                if (srcLine.IndexOf("<Acid>") >= 0 && dstLine.IndexOf("<Acid>") >= 0)
                {
                    continue;
                }
                else if (srcLine != dstLine)
                {
                    break;
                }
            }

            bool result = srcReader.EndOfStream && dstReader.EndOfStream;
            src.Seek(srcCurrentOffset, SeekOrigin.Begin);
            dst.Seek(dstCurrentOffset, SeekOrigin.Begin);
            if (result)
            {
                // ファイルの内容が等しくても、src に署名があり、dst には署名がない場合には更新する必要がある。
                // そのため、この場合においては、文面に違いがあるとして結果を返す
                result = !(srcHasKey && !dstHasKey);
            }
            return result;
        }

        static void serializer_UnknownElement(object sender, XmlElementEventArgs e)
        {
            throw new ArgumentException(string.Format(Properties.Resources.Message_UnknownElement, e.Element.LocalName));
        }
    }
}
