﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------namespace MakeDesc

using System;
using System.Collections.Generic;
using System.Security.Cryptography;

namespace MakeDesc
{
    internal class DerRsaPrivateKey
    {
        internal RSAParameters RsaPrivateKey { get; private set; }

        internal int Version { get; private set; }
        internal DerUniversalInteger Modulus { get; private set; }
        internal DerUniversalInteger PublicExponent { get; private set; }
        internal DerUniversalInteger PrivateExponent { get; private set; }
        internal DerUniversalInteger Prime1 { get; private set; }
        internal DerUniversalInteger Prime2 { get; private set; }
        internal DerUniversalInteger Exponent1 { get; private set; }
        internal DerUniversalInteger Exponent2 { get; private set; }
        internal DerUniversalInteger Coefficient { get; private set; }

        internal List<DerRsaOtherPrimeInfo> OtherPrimeInfos { get; private set; }

        internal DerRsaPrivateKey(byte[] derData)
        {
            DerUniversalSequence header = new DerUniversalSequence(derData, 0);
            if (header.ConsumedBytes != derData.Length)
            {
                throw new FormatException(Properties.Resources.Message_FailToParseKey);
            }

            int index = 0;
            byte[] data = header.Contents;

            DerUniversalInteger version = new DerUniversalInteger(data, index);
            index += version.ConsumedBytes;
            if (version.Contents.Length != 1)
            {
                throw new FormatException(Properties.Resources.Message_FailToParseKey);
            }

            this.Version = version.Contents[0];
            RSAParameters key = new RSAParameters();

            // RSA 2048 向け鍵長チェック
            DerUniversalInteger modulus = new DerUniversalInteger(data, index);
            index += modulus.ConsumedBytes;
            key.Modulus = CheckKeySizeAndSkipSign(modulus.Contents, 0x100);

            DerUniversalInteger publicExponent = new DerUniversalInteger(data, index);
            index += publicExponent.ConsumedBytes;
            key.Exponent = CheckKeySizeAndSkipSign(publicExponent.Contents, 0x3);

            DerUniversalInteger privateExponent = new DerUniversalInteger(data, index);
            index += privateExponent.ConsumedBytes;
            key.D = CheckKeySizeAndSkipSign(privateExponent.Contents, 0x100);

            DerUniversalInteger prime1 = new DerUniversalInteger(data, index);
            index += prime1.ConsumedBytes;
            key.P = CheckKeySizeAndSkipSign(prime1.Contents, 0x80);

            DerUniversalInteger prime2 = new DerUniversalInteger(data, index);
            index += prime2.ConsumedBytes;
            key.Q = CheckKeySizeAndSkipSign(prime2.Contents, 0x80);

            DerUniversalInteger exponent1 = new DerUniversalInteger(data, index);
            index += exponent1.ConsumedBytes;
            key.DP = CheckKeySizeAndSkipSign(exponent1.Contents, 0x80);

            DerUniversalInteger exponent2 = new DerUniversalInteger(data, index);
            index += exponent2.ConsumedBytes;
            key.DQ = CheckKeySizeAndSkipSign(exponent2.Contents, 0x80);

            DerUniversalInteger coefficient = new DerUniversalInteger(data, index);
            index += coefficient.ConsumedBytes;
            key.InverseQ = CheckKeySizeAndSkipSign(coefficient.Contents, 0x80);

            if (version.Contents[0] == 1)
            {
                DerUniversalSequence otherInfos = new DerUniversalSequence(data, index);
                byte[] otherData = otherInfos.Contents;
                int otherSize = otherInfos.ConsumedBytes;

                int consume = 0;
                while (consume < otherSize)
                {
                    DerRsaOtherPrimeInfo info = new DerRsaOtherPrimeInfo(data, index);
                    index += info.ConsumedBytes;
                    consume += info.ConsumedBytes;

                    this.OtherPrimeInfos.Add(info);
                }

                if (consume != otherSize)
                {
                    throw new FormatException(Properties.Resources.Message_FailToParseKey);
                }
            }

            if (index != header.Contents.Length)
            {
                throw new FormatException(Properties.Resources.Message_FailToParseKey);
            }

            this.RsaPrivateKey = key;
        }

        private byte[] CheckKeySizeAndSkipSign(byte[] src, int size)
        {
            int offset = 0;
            int length = src.Length;
            // openssl genrsa の符号削除
            if (length == size + 1)
            {
                if (src[0] == 0)
                {
                    offset++;
                    length--;
                }
            }
            else if (length != size)
            {
                throw new FormatException(Properties.Resources.Message_FailToParseKey);
            }
            byte[] dst = new byte[length];
            Array.Copy(src, offset, dst, 0, length);
            return dst;
        }
    }

    internal class DerRsaOtherPrimeInfo
    {
        internal DerUniversalSequence Header { get; private set; }
        internal DerUniversalInteger Prime;
        internal DerUniversalInteger Exponent;
        internal DerUniversalInteger Coefficient;
        internal int ConsumedBytes { get; private set; }

        internal DerRsaOtherPrimeInfo(byte[] data, int startOffset)
        {
            Header = new DerUniversalSequence(data, startOffset);

            int index = 0;

            this.Prime = new DerUniversalInteger(Header.Contents, index);
            index += this.Prime.ConsumedBytes;

            this.Exponent = new DerUniversalInteger(data, startOffset + this.ConsumedBytes);
            index += this.Exponent.ConsumedBytes;

            this.Coefficient = new DerUniversalInteger(data, startOffset + this.ConsumedBytes);
            index  = this.Coefficient.ConsumedBytes;

            if (index != Header.Contents.Length)
            {
                throw new FormatException(Properties.Resources.Message_FailToParseKey);
            }

            this.ConsumedBytes = Header.ConsumedBytes;
        }
    }
}
