﻿using System;
using System.IO;
using System.Collections.Specialized;

namespace MakeDesc
{
    internal class DerId
    {
        BitVector32 IdVec;
        BitVector32.Section IdTag;
        BitVector32.Section IdFlag;
        BitVector32.Section IdClass;

        internal enum IdClassType
        {
            Universal,
            Unknown,
        }

        internal const bool IdFlagComplex = true;
        internal const bool IdFlagPrimitive = false;

        internal enum IdTagType
        {
            Sequence,
            Integer,
            Unknown,
        }

        internal DerId(byte d)
        {
            IdVec = new BitVector32(d);
            IdTag = BitVector32.CreateSection((1 << 5) - 1);
            IdFlag = BitVector32.CreateSection(1, IdTag);
            IdClass = BitVector32.CreateSection((1 << 2) - 1, IdFlag);
        }

        public IdTagType Tag
        {
            get
            {
                switch (IdVec[IdTag])
                {
                    case 0x10: return IdTagType.Sequence;
                    case 0x02: return IdTagType.Integer;
                    default: return IdTagType.Unknown;
                }
            }
        }

        public bool Flag
        {
            get
            {
                return IdVec[IdFlag] != 0;
            }
        }

        public IdClassType ClassType
        {
            get
            {
                switch(IdVec[IdClass])
                {
                    case 0: return IdClassType.Universal;
                    default: return IdClassType.Unknown;
                }
            }
        }

        public bool Check(IdClassType idClass, bool flag, IdTagType tag)
        {
            if (this.ClassType != idClass)
            {
                return false;
            }
            if (this.Flag != flag)
            {
                return false;
            }
            if (this.Tag != tag)
            {
                return false;
            }
            return true;
        }

    }

    internal class DerLength
    {
        BitVector32 LengthVec;
        BitVector32.Section LengthFlag;
        BitVector32.Section LengthData;

        internal DerLength(byte d)
        {
            LengthVec = new BitVector32(d);
            LengthData = BitVector32.CreateSection((1 << 7) - 1);
            LengthFlag = BitVector32.CreateSection(1, LengthData);
        }

        internal bool HasContents
        {
            get
            {
                return LengthVec[LengthFlag] == 0;
            }
        }

        internal byte Data
        {
            get
            {
                return (byte)LengthVec[LengthData];
            }
        }

        internal int ReadLength(out int length, byte[] data, int offset)
        {
            if (this.HasContents)
            {
                length = this.Data;
                return 0;
            }

            if (this.Data > 4)
            {
                throw new FormatException(string.Format(Properties.Resources.Message_InvalidDerSize, this.Data));
            }

            length = 0;
            for (int i = 0; i < this.Data; i++)
            {
                length = (length << (i * 8)) | data[offset + i];
            }

            return this.Data;
        }
    }

    internal class DerUniversalSequence
    {
        internal DerId Id { get; private set; }
        internal DerLength Length { get; private set; }
        internal byte[] Contents { get; private set; }
        internal int ConsumedBytes { get; private set; }

        internal DerUniversalSequence(byte[] data, int startOffset)
        {
            this.ConsumedBytes = 0;
            Id = new DerId(data[this.ConsumedBytes++]);

            if (!Id.Check(DerId.IdClassType.Universal, DerId.IdFlagComplex, DerId.IdTagType.Sequence))
            {
                throw new FormatException(string.Format(Properties.Resources.Message_InvalidDerClass, "Universal", "Sequence"));
            }

            DerLength lengthField = new DerLength(data[this.ConsumedBytes++]);

            int length;
            this.ConsumedBytes += lengthField.ReadLength(out length, data, this.ConsumedBytes);

            this.Contents = new byte[length];
            Array.Copy(data, this.ConsumedBytes, this.Contents, 0, length);
            this.ConsumedBytes += length;
        }
    }

    internal class DerUniversalInteger
    {
        internal DerId Id { get; private set; }
        internal DerLength Length { get; private set; }
        internal byte[] Contents { get; private set; }
        internal int ConsumedBytes { get; private set; }

        internal DerUniversalInteger(byte[] data, int startOffset)
        {
            this.ConsumedBytes = 0;
            this.Id = new DerId(data[startOffset + this.ConsumedBytes]);
            this.ConsumedBytes++;

            if (!this.Id.Check(DerId.IdClassType.Universal, DerId.IdFlagPrimitive, DerId.IdTagType.Integer))
            {
                throw new FormatException(string.Format(Properties.Resources.Message_InvalidDerClass, "Universal", "Integer"));
            }

            this.Length = new DerLength(data[startOffset + this.ConsumedBytes]);
            this.ConsumedBytes++;

            int length;
            this.ConsumedBytes += this.Length.ReadLength(out length, data, startOffset + this.ConsumedBytes);
            this.Contents = new byte[length];
            Array.Copy(data, startOffset + this.ConsumedBytes, this.Contents, 0, length);
            this.ConsumedBytes += length;
        }
    }
}
