﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Xml;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices;
using System.Security.Cryptography;

namespace MakeDesc
{
    [StructLayout(LayoutKind.Sequential)]
    internal struct AcidHeader
    {
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x100)]
        public byte[] HeaderSignature;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x100)]
        public byte[] HeaderDecryptKey;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x4)]
        public byte[] Signature;
        public uint SignatureAreaSize;
        public byte descVersion;
        public byte Reserved1_1;
        public byte Reserved1_2;
        public byte Reserved1_3;
        public uint flags;
        public ulong ProgramIdMin;
        public ulong ProgramIdMax;
        public uint FacdOffset;
        public uint FacdSize;
        public uint SacdOffset;
        public uint SacdSize;
        public uint KcdOffset;
        public uint KcdSize;
        public ulong Reserved2;
    }

    /// <summary>
    /// ACID を管理するクラス:w
    /// </summary>
    public class Acid
    {
        private AcidHeader header;
        private const string Signature = "ACID";
        private const string AcidNodeName = "Acid";

        // flags 用のビット定義
        private const uint ProductionFlagMask      = 0x01;
        private const uint UnqualifiedApprovalMask = 0x02;
        private const int  UnqualifiedApprovalShift = 1;
        private const uint MemoryRegionMask = 0x3C;
        private const int  MemoryRegionShift = 2;

        public byte[] FsAccessControlDescriptor { get; set; }
        public byte[] SrvAccessControlDescriptor { get; set; }
        public uint[] KernelCapabilityDescriptor { get; set; }

        public ulong ProgramIdMax
        {
            get
            {
                return header.ProgramIdMax;
            }
            set
            {
                header.ProgramIdMax = value;
            }
        }

        public ulong ProgramIdMin
        {
            get
            {
                return header.ProgramIdMin;
            }
            set
            {
                header.ProgramIdMin = value;
            }
        }

        public FacDescriptor Facd { get; set; }
        public SacDescriptor Sacd { get; set; }
        public KcDescriptor Kcd { get; set; }
        public bool ProductionFlag
        {
            get { return Convert.ToBoolean(header.flags & ProductionFlagMask); }
            set { header.flags |= (header.flags & ~ProductionFlagMask) | (value ? 1u : 0u); }
        }

        public bool UnqualifiedApproval
        {
            get { return Convert.ToBoolean(header.flags & UnqualifiedApprovalMask); }
            set { header.flags |= (header.flags & ~UnqualifiedApprovalMask) | ((value ? 1u : 0u) << UnqualifiedApprovalShift); }
        }

        public byte MemoryRegion
        {
            get { return (byte)((header.flags & MemoryRegionMask) >> MemoryRegionShift); }
            set { header.flags |= (header.flags & ~MemoryRegionMask) | (uint)((value << MemoryRegionShift) & MemoryRegionMask); }
        }

        public byte DescVersion
        {
            get { return header.descVersion; }
            set { header.descVersion = value; }
        }

        public Acid()
        {
            header = new AcidHeader();
            header.Signature = System.Text.Encoding.ASCII.GetBytes(Signature);
            header.flags = 0x0;
        }

        public byte[] ExportAcidBinary(KeyManager key)
        {
            uint headerSize = (uint)Marshal.SizeOf(typeof(AcidHeader));

            byte[] facdBinary = Facd.ExportBinary();
            byte[] sacdBinary = Sacd.ExportBinary();
            byte[] kcdBinary = Kcd.ExportBinary();

            uint alignSize = 0x10;
            header.FacdOffset = headerSize;
            header.FacdSize = (uint)((facdBinary != null) ? facdBinary.Length : 0);
            header.SacdOffset = header.FacdOffset + RoundUp(header.FacdSize, alignSize);
            header.SacdSize = (uint)((sacdBinary != null) ? sacdBinary.Length : 0);

            header.KcdOffset = header.SacdOffset + RoundUp(header.SacdSize, alignSize);
            header.KcdSize = (uint)((kcdBinary != null) ? kcdBinary.Length : 0);

            uint facdBinSize = (header.SacdSize > 0 || header.KcdSize > 0) ? RoundUp(header.FacdSize, alignSize) : header.FacdSize;
            uint sacdBinSize = (header.KcdSize > 0) ? RoundUp(header.SacdSize, alignSize) : header.SacdSize;
            uint kcdBinSize = header.KcdSize;

            uint binarySize = headerSize + facdBinSize + sacdBinSize + kcdBinSize;
            byte[] outputBinary = new byte[binarySize];

            header.SignatureAreaSize = binarySize - 0x100;

            if (header.FacdSize > 0)
            {
                Array.Copy(facdBinary, 0, outputBinary, header.FacdOffset, header.FacdSize);
            }

            if (header.SacdSize > 0)
            {
                Array.Copy(sacdBinary, 0, outputBinary, header.SacdOffset, header.SacdSize);
            }

            if (header.KcdSize > 0)
            {
                Array.Copy(kcdBinary, 0, outputBinary, header.KcdOffset, header.KcdSize);
            }

            if (key.HasKey())
            {
                if (key.HeaderKey.Modulus.Length > 0x100)
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidSignatureHeaderKeySize, key.HeaderKey.Modulus.Length));
                }
                header.HeaderDecryptKey = new byte[0x100];
                Array.Copy(key.HeaderKey.Modulus, header.HeaderDecryptKey, 0x100);
            }

            GCHandle gch = GCHandle.Alloc(outputBinary, GCHandleType.Pinned);
            Marshal.StructureToPtr(header, gch.AddrOfPinnedObject(), false);
            gch.Free();

            if (key.HasKey())
            {
                Nintendo.MakeDesc.CryptoLibrary.Rsa2048PssSha256SignCryptoDriver rsa = new Nintendo.MakeDesc.CryptoLibrary.Rsa2048PssSha256SignCryptoDriver();

                byte[] signData = rsa.SignBlock(key.PrivateKey.Modulus, key.PrivateKey.D, outputBinary, 0x100, (int)header.SignatureAreaSize);
                if (signData.Length != 0x100)
                {
                    throw new ArgumentException(Properties.Resources.Message_FaildToSign);
                }

                Array.Copy(signData, outputBinary, 0x100);
            }


            return outputBinary;
        }

        public uint RoundUp(uint target, uint align)
        {
            return ((target + (align - 1)) & ~(align - 1));
        }
    }
}
