﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using Nintendo.Foundation.IO;

namespace MakeAddrRegionHeader
{
    internal class Program
    {
        private const string ProgramDescription = "アドレス領域を定義するヘッダファイルを生成します。";

        private static void Main(string[] args)
        {
#if !DEBUG
            try
#endif
            {
                MakeAddrRegionHeaderParam param;

                try
                {
                    param = ParseArgs(args);
                    if (param == null)
                    {
                        // ヘルプオプションが指定されたら、終了
                        return;
                    }
                }
                catch
                {
                    Environment.ExitCode = 1;
                    return;
                }

                var maker = new AddrRegionHeaderMaker(param);
                maker.MakeAddrRegionHeader();

                Environment.ExitCode = maker.IsError ? 1 : 0;
            }
            // Release ビルドの場合、ハンドルされていない例外をキャッチして、終了コード 1 で終了します。
            // Debug ビルドの場合はデバッガでハンドルできるように、ハンドルされていないままにします。
#if !DEBUG
            catch (Exception e)
            {
                Console.WriteLine(e.ToString());
                Environment.ExitCode = 1;
                return;
            }
#endif
        }

        // テストするために internal にしておく
        internal static MakeAddrRegionHeaderParam ParseArgs(string[] args)
        {
            var settings = new CommandLineParserSettings()
            {
                ApplicationDescription = ProgramDescription,
                ErrorWriter = text => Logger.WriteCommandLineError(text),
            };

            MakeAddrRegionHeaderParam result = null;
            new CommandLineParser(settings).ParseArgs(args, out result);

            return result;
        }
    }

    internal class MakeAddrRegionHeaderParam
    {
        // ValueName を日本語で指定した場合、CommandLineParser.CreateHelpText の桁揃えがずれます。
        // 回避するため、英語で指定します。

        [CommandLineOption(
            "input",
            IsRequired = true,
            ValueName = "path",
            Description = "入力ファイルを指定します。")]
        public string InputFilePath { get; private set; }

        [CommandLineOption(
            "output",
            IsRequired = true,
            ValueName = "path",
            Description = "出力ファイルを指定します。")]
        public string OutputFilePath { get; private set; }

        [CommandLineOption(
            "64",
            DefaultValue = false,
            IsRequired = false,
            Description = "出力を 64 bit 幅で行います。指定しない場合、出力を 32 bit 幅で行います。")]
        public bool Is64BitMode { get; private set; }

        [CommandLineOption(
            "prefix",
            DefaultValue = "",
            IsRequired = false,
            ValueName = "prefix",
            Description = "マクロ名に付加するプレフィックス文字列を指定します。指定しない場合は空文字列が指定されたものとして扱います。")]
        public string MacroNamePrefix { get; private set; }

        [CommandLineOption(
            "address-convert-macro",
            DefaultValue = "",
            IsRequired = false,
            ValueName = "macro name",
            Description = "アドレス変換マクロ名を指定します。指定しない場合は空文字列が指定されたものとして扱います。")]
        public string ConvertMacroName { get; private set; }
    }

    internal class AddrRegionHeaderMaker
    {
        public bool IsError { get; private set; }

        private string inputFilePath;
        private string outputFilePath;
        private string macroNamePrefix;
        private string convertMacroName;
        private AddrWidth addrWidth;

        public AddrRegionHeaderMaker(MakeAddrRegionHeaderParam param)
        {
            this.inputFilePath = param.InputFilePath;
            this.outputFilePath = param.OutputFilePath;
            this.macroNamePrefix = param.MacroNamePrefix;
            this.convertMacroName = param.ConvertMacroName;
            this.addrWidth = param.Is64BitMode ? AddrWidth.Bit64 : AddrWidth.Bit32;
        }

        public void MakeAddrRegionHeader()
        {
            var reader = new AddrRegionDefinitionReader(addrWidth);
            var statements = reader.Read(inputFilePath);

            var evaluator = new Evaluator();
            var values = evaluator.Evaluate(statements);

            // エラーがあったら出力せず終了
            if (reader.IsError || evaluator.IsError)
            {
                this.IsError = true;
                return;
            }

            var writer = new AddrRegionHeaderWriter(macroNamePrefix, convertMacroName);
            writer.Write(outputFilePath, values);
        }
    }

    internal class AddrRegionDefinitionReader
    {
        public bool IsError { get; private set; }
        private readonly AddrWidth addrWidth;

        public AddrRegionDefinitionReader(AddrWidth addrWidth)
        {
            this.addrWidth = addrWidth;
        }

        public IEnumerable<Statement> Read(string path)
        {
            List<Statement> statements = new List<Statement>();

            foreach (var line in EnumerateLines(path))
            {
                var context = new ParseContext(addrWidth, new FilePosition(path, line.LineNumber));
                try
                {
                    var statement = StatementParser.Parse(line.Text, context);

                    // include を特別扱いする
                    if (statement is IncludeStatement)
                    {
                        var includeStatement = (IncludeStatement)statement;
                        string includedPath = PathUtil.GetRelativePath(path, includeStatement.FileName);
                        statements.AddRange(Read(includedPath));
                    }
                    else
                    {
                        statements.Add(statement);
                    }
                }
                catch (MakeAddrRegionHeaderException e)
                {
                    Logger.WriteError(context.Position, e);
                    this.IsError = true;
                }
            }

            return statements;
        }

        private static IEnumerable<Line> EnumerateLines(string path)
        {
            int lineNumber = 1;
            foreach (var lineText in File.ReadAllLines(path))
            {
                yield return new Line(lineText, lineNumber);
                lineNumber++;
            }
        }

        private class Line
        {
            public string Text { get; private set; }
            public int LineNumber { get; private set; }

            public Line(string text, int lineNumber)
            {
                this.Text = text;
                this.LineNumber = lineNumber;
            }
        }
    }

    internal class AddrRegionHeaderWriter
    {
        private readonly string macroNamePrefix;
        private readonly string convertMacroName;

        public AddrRegionHeaderWriter(string macroNamePrefix, string convertMacroName)
        {
            this.macroNamePrefix = macroNamePrefix;
            this.convertMacroName = convertMacroName;
        }

        public void Write(string path, IEnumerable<ISymbolEmitter> elements)
        {
            using (StreamWriter sw = new StreamWriter(path, false, Encoding.UTF8))
            {
                foreach (var element in elements)
                {
                    WriteAddrElement(sw, element);
                }
            }
        }

        private void WriteAddrElement(StreamWriter sw, ISymbolEmitter element)
        {
            var symbols = element.EnumerateSymbols().ToList();
            int maxSymbolNameLength = symbols.Select(s => s.Name.Length).Max();

            foreach (var symbol in symbols)
            {
                sw.WriteLine("#define {0} {1}",
                    MakeMacroName(symbol.Name.PadRight(maxSymbolNameLength)), MakeMacroValue(symbol.Value));
            }
            sw.WriteLine();
        }

        private string MakeMacroValue(AddrValue value)
        {
            return string.Format(CultureInfo.InvariantCulture, "{0}(0x{1})",
                MakeMacroName(convertMacroName), value.ToString());
        }

        private string MakeMacroName(string name)
        {
            return macroNamePrefix + name;
        }
    }
}
