﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeAddrRegionHeader
{
    internal class Evaluator
    {
        public bool IsError { get; private set; }
        private EvaluationContext context;

        public Evaluator()
        {
            this.context = new EvaluationContext();
        }

        public IEnumerable<ISymbolEmitter> Evaluate(IEnumerable<Statement> statements)
        {
            foreach (var statement in statements)
            {
                try
                {
                    statement.Evaluate(context);
                }
                catch (MakeAddrRegionHeaderException e)
                {
                    this.IsError = true;
                    Logger.WriteError(statement.Position, e);
                }
            }

            return context.GetSymbolEmitters();
        }
    }

    internal class EvaluationContext
    {
        private SymbolTable symbolTable = new SymbolTable();
        private Dictionary<string, AddrRegion> addrRegions = new Dictionary<string, AddrRegion>();

        public void AddAddrRegion(AddrRegion region)
        {
            if (addrRegions.ContainsKey(region.Name))
            {
                throw new DuplicatedNameException(
                    string.Format(CultureInfo.CurrentCulture, "領域 {0} は既に定義されています。", region.Name));
            }

            symbolTable.AddSymbolEmitter(region);
            addrRegions.Add(region.Name, region);
        }

        public AddrRegion GetAddrRegion(string regionName)
        {
            if (!addrRegions.ContainsKey(regionName))
            {
                throw new UndefinedNameException(
                    string.Format(CultureInfo.CurrentCulture, "領域 {0} は定義されていません。", regionName));
            }

            return addrRegions[regionName];
        }

        public void AddAddrConstant(AddrConstant constant)
        {
            symbolTable.AddSymbolEmitter(constant);
        }

        public IEnumerable<ISymbolEmitter> GetSymbolEmitters()
        {
            return symbolTable.GetSymbolEmitters();
        }
    }

    internal class SymbolTable
    {
        private List<ISymbolEmitter> emitters = new List<ISymbolEmitter>();
        private HashSet<string> definedSymbolName = new HashSet<string>();

        public void AddSymbolEmitter(ISymbolEmitter emitter)
        {
            emitters.Add(emitter);

            foreach (var symbol in emitter.EnumerateSymbols())
            {
                AddDefinedName(symbol.Name);
            }
        }

        private void AddDefinedName(string name)
        {
            if (!definedSymbolName.Add(name))
            {
                throw new DuplicatedNameException(
                    string.Format(CultureInfo.CurrentCulture, "シンボル {0} は既に定義されています。", name));
            }
        }

        public IEnumerable<ISymbolEmitter> GetSymbolEmitters()
        {
            return emitters;
        }
    }
}
