﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Globalization;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace MakeAddrRegionHeader
{
    internal enum InnerOuter
    {
        Inner,
        Outer
    }

    internal enum RegionValueType
    {
        Begin,
        End,
        Size
    }

    internal enum ArgumentType
    {
        InnerOuter,
        Shift,
        Alias,
        Include,
        RegionValueType,
        Identifier,
        StringLiteral,
        ConstantSpecifier,
        OffsetSpecifier,
        SizeSpecifier,
        BeginSpecifier,
        EndSpecifier,
    }

    internal class ArgumentUtil
    {
        private static readonly Regex ConstantSpecifierRegex = new Regex(@"^=(.*)$", RegexOptions.Compiled);
        private static readonly Regex OffsetSpecifierRegex = new Regex(@"^(\+|\-)(.*)$", RegexOptions.Compiled);
        private static readonly Regex SizeSpecifierRegex = new Regex(@"^size=(.*)$", RegexOptions.Compiled);
        private static readonly Regex BeginSpecifierRegex = new Regex(@"^begin=(.*)$", RegexOptions.Compiled);
        private static readonly Regex EndSpecifierRegex = new Regex(@"^end=(.*)$", RegexOptions.Compiled);
        private static readonly Regex StringLiteralRegex = new Regex("^\"([^\"]*)\"$", RegexOptions.Compiled);
        private static readonly Regex IdentifierRegex = new Regex(@"^[_a-zA-Z][_a-zA-Z0-9]+$", RegexOptions.Compiled);

        // 上ほどマッチの優先度が高い（予約語を上に書く）
        private static readonly List<ArgumentPattern> ArgumentPatterns = new List<ArgumentPattern>()
        {
            new ArgumentPattern(ArgumentType.InnerOuter,           new Regex(@"^(inner|outer)$", RegexOptions.Compiled)),
            new ArgumentPattern(ArgumentType.RegionValueType,      new Regex(@"^(begin|end|size)$", RegexOptions.Compiled)),
            new ArgumentPattern(ArgumentType.Include,              new Regex(@"^include$", RegexOptions.Compiled)),
            new ArgumentPattern(ArgumentType.Shift,                new Regex(@"^shift$", RegexOptions.Compiled)),
            new ArgumentPattern(ArgumentType.Alias,                new Regex(@"^alias$", RegexOptions.Compiled)),
            new ArgumentPattern(ArgumentType.ConstantSpecifier,    ConstantSpecifierRegex),
            new ArgumentPattern(ArgumentType.OffsetSpecifier,      OffsetSpecifierRegex),
            new ArgumentPattern(ArgumentType.SizeSpecifier,        SizeSpecifierRegex),
            new ArgumentPattern(ArgumentType.BeginSpecifier,       BeginSpecifierRegex),
            new ArgumentPattern(ArgumentType.EndSpecifier,         EndSpecifierRegex),
            new ArgumentPattern(ArgumentType.StringLiteral,        StringLiteralRegex),
            new ArgumentPattern(ArgumentType.Identifier,           IdentifierRegex),
        };

        public static ArgumentType GetArgumentType(string argumentString)
        {
            try
            {
                return ArgumentPatterns.First(p => p.IsMatch(argumentString)).ArgumentType;
            }
            catch (InvalidOperationException)
            {
                throw new SyntaxErrorException(
                    string.Format(CultureInfo.CurrentCulture, "{0} : 引数の形式が不正です。", argumentString));
            }
        }

        public static InnerOuter ParseInnerOuter(string s)
        {
            switch (s)
            {
                case "inner":
                    return InnerOuter.Inner;
                case "outer":
                    return InnerOuter.Outer;
                default:
                    throw new SyntaxErrorException(
                        string.Format(CultureInfo.CurrentCulture, "{0} は、innner、outer のいずれかである必要があります。", s));
            }
        }

        public static RegionValueType ParseRegionValueType(string s)
        {
            switch (s)
            {
                case "begin":
                    return RegionValueType.Begin;
                case "end":
                    return RegionValueType.End;
                case "size":
                    return RegionValueType.Size;
                default:
                    throw new SyntaxErrorException(
                        string.Format(CultureInfo.CurrentCulture, "{0} は、begin、end、size のいずれかである必要があります。", s));
            }
        }

        public static AddrValue ParseConstantSpecifier(string s, AddrWidth addrWidth)
        {
            return ParseNumberSpecifier(s, addrWidth, ConstantSpecifierRegex, "=<数値>");
        }

        public static AddrValue ParseSizeSpecifier(string s, AddrWidth addrWidth)
        {
            return ParseNumberSpecifier(s, addrWidth, SizeSpecifierRegex, "size=<数値>");
        }

        public static AddrValue ParseBeginSpecifier(string s, AddrWidth addrWidth)
        {
            return ParseNumberSpecifier(s, addrWidth, BeginSpecifierRegex, "begin=<数値>");
        }

        public static AddrValue ParseEndSpecifier(string s, AddrWidth addrWidth)
        {
            return ParseNumberSpecifier(s, addrWidth, EndSpecifierRegex, "end=<数値>");
        }

        private static AddrValue ParseNumberSpecifier(string argumentString, AddrWidth addrWidth, Regex formatRe, string formatString)
        {
            var match = formatRe.Match(argumentString);
            if (!match.Success)
            {
                throw new SyntaxErrorException(
                    string.Format(CultureInfo.CurrentCulture, "{0} は、{1} の形式である必要があります。", argumentString, formatString));
            }

            return ParseUtil.ParseAddrValue(match.Groups[1].Value, addrWidth);
        }

        public static Offset ParseOffsetSpecifier(string argumentString, AddrWidth addrWidth)
        {
            var match = OffsetSpecifierRegex.Match(argumentString);
            if (!match.Success)
            {
                throw new SyntaxErrorException(
                    string.Format(CultureInfo.CurrentCulture, "{0} は、(+|-)<数値> の形式である必要があります。", argumentString));
            }

            return new Offset(
                ParseOffsetSign(match.Groups[1].Value), ParseUtil.ParseAddrValue(match.Groups[2].Value, addrWidth));
        }

        private static Sign ParseOffsetSign(string s)
        {
            switch (s)
            {
                case "+":
                    return Sign.Positive;
                case "-":
                    return Sign.Negative;
                default:
                    throw new SyntaxErrorException(
                        string.Format(CultureInfo.CurrentCulture, "{0} は、+、-、のいずれかである必要があります。", s));
            }
        }

        public static string ParseAsStringLiteral(string argumentString)
        {
            var match = StringLiteralRegex.Match(argumentString);
            if (!match.Success)
            {
                throw new SyntaxErrorException(
                    string.Format(CultureInfo.CurrentCulture, "{0} は、\"文字列\" の形式である必要があります。", argumentString));
            }

            return match.Groups[1].Value;
        }

        public static string ParseAsIdentifier(string s)
        {
            if (!IdentifierRegex.IsMatch(s))
            {
                throw new SyntaxErrorException(
                    string.Format(CultureInfo.CurrentCulture, "{0} は、<名前>として無効です。", s));
            }

            return s;
        }

        private class ArgumentPattern
        {
            public ArgumentType ArgumentType { get; private set; }
            private Regex pattern;

            public ArgumentPattern(ArgumentType argumentType, Regex pattern)
            {
                this.ArgumentType = argumentType;
                this.pattern = pattern;
            }

            public bool IsMatch(string s)
            {
                return pattern.IsMatch(s);
            }
        }
    }
}
