﻿using System;
using System.Collections.Generic;
using System.Numerics;
using System.Text.RegularExpressions;

namespace Nintendo.Log
{
    internal class LogFilterParserNode
    {
        public LogFilterParserNode(object lhs, object rhs, OpcodeType opcode)
        {
            Lhs = lhs;
            Rhs = rhs;
            Opcode = opcode;
        }

        public object Evaluate(Dictionary<string, object> variableTable)
        {
            if (Opcode == OpcodeType.None)
            {
                if (Rhs is FilterVariable)
                {
                    var variableName = ((FilterVariable)Rhs).Name;
                    return variableTable.ContainsKey(variableName)
                        ? variableTable[variableName]
                        : null;
                }
                else
                {
                    return Rhs;
                }
            }
            else if (Opcode == OpcodeType.Not)
            {
                var rhs = (Rhs is LogFilterParserNode)
                    ? ((LogFilterParserNode)Rhs).Evaluate(variableTable)
                    : Rhs;
                if (rhs is bool)
                {
                    return !((bool)rhs);
                }
                else
                {
                    return false;
                }
            }
            else if (Opcode == OpcodeType.Match || Opcode == OpcodeType.NotMatch)
            {
                var lhs = (Lhs is LogFilterParserNode)
                    ? ((LogFilterParserNode)Lhs).Evaluate(variableTable)
                    : Lhs;
                var rhs = (Rhs is LogFilterParserNode)
                    ? ((LogFilterParserNode)Rhs).Evaluate(variableTable)
                    : Rhs;
                if (lhs is string && rhs is Regex)
                {
                    return EvaluateComparison((string)lhs, (Regex)rhs, Opcode);
                }
                else if (lhs is Regex && rhs is string)
                {
                    return EvaluateComparison((string)rhs, (Regex)lhs, Opcode);
                }
                else
                {
                    return false;
                }
            }
            else
            {
                var lhs = (Lhs is LogFilterParserNode)
                    ? ((LogFilterParserNode)Lhs).Evaluate(variableTable)
                    : Lhs;
                var rhs = (Rhs is LogFilterParserNode)
                    ? ((LogFilterParserNode)Rhs).Evaluate(variableTable)
                    : Rhs;
                if (lhs == null && rhs == null)
                {
                    return false;
                }
                else if (!CheckComparisonType(lhs, rhs))
                {
                    return false;
                }
                else if (lhs.IsInteger())
                {
                    return EvaluateComparison(lhs.ToBigInteger(), rhs.ToBigInteger(), Opcode);
                }
                else if (lhs is bool)
                {
                    return EvaluateComparison((bool)lhs, (bool)rhs, Opcode);
                }
                else if (lhs is string)
                {
                    return EvaluateComparison((string)lhs, (string)rhs, Opcode);
                }
                else
                {
                    return false;
                }
            }
        }

        public bool EvaluateComparison(BigInteger lhs, BigInteger rhs, OpcodeType opcode)
        {
            switch (opcode)
            {
                case OpcodeType.GreaterThan:
                    return lhs > rhs;
                case OpcodeType.LessThan:
                    return lhs < rhs;
                case OpcodeType.GreaterEqual:
                    return lhs >= rhs;
                case OpcodeType.LessEqual:
                    return lhs <= rhs;
                case OpcodeType.Equal:
                    return lhs == rhs;
                case OpcodeType.NotEqual:
                    return lhs != rhs;
                default:
                    throw new FilterSyntaxErrorException("Operators can not be applied to the comparison of the integer.");
            }
        }

        public bool EvaluateComparison(bool lhs, bool rhs, OpcodeType opcode)
        {
            switch (opcode)
            {
                case OpcodeType.Equal:
                    return (lhs == rhs);
                case OpcodeType.NotEqual:
                    return (lhs != rhs);
                case OpcodeType.And:
                    return (lhs && rhs);
                case OpcodeType.Or:
                    return (lhs || rhs);
                default:
                    throw new FilterSyntaxErrorException("Operators can not be applied to the comparison of the boolean.");
            }
        }

        public bool EvaluateComparison(string lhs, string rhs, OpcodeType opcode)
        {
            switch (opcode)
            {
                case OpcodeType.Equal:
                    return (lhs == rhs);
                case OpcodeType.NotEqual:
                    return (lhs != rhs);
                default:
                    throw new FilterSyntaxErrorException("Operators can not be applied to the comparison of the string");
            }
        }

        public bool EvaluateComparison(string lhs, Regex rhs, OpcodeType opcode)
        {
            if (opcode == OpcodeType.Match)
            {
                return rhs.Match(lhs).Success;
            }
            else
            {
                return !rhs.Match(lhs).Success;
            }
        }

        private static bool CheckComparisonType(object lhs, object rhs)
        {
            if (lhs.IsInteger())
            {
                return rhs.IsInteger();
            }
            else
            {
                return (lhs?.GetType() == rhs?.GetType());
            }
        }

        public object Lhs { get; set; }
        public object Rhs { get; set; }
        public OpcodeType Opcode { get; set; }
    }

    internal static class ObjectExtention
    {
        public static bool IsInteger(this object value)
        {
            return (value.IsSignedInteger() || value.IsUnsignedInteger());
        }

        public static bool IsSignedInteger(this object value)
        {
            return (value is sbyte || value is short || value is int || value is long);
        }

        public static bool IsUnsignedInteger(this object value)
        {
            return (value is byte || value is ushort || value is uint || value is ulong);
        }

        private static Dictionary<Type, Func<object, BigInteger>> ToBigIntegerTable =
            new Dictionary<Type, Func<object, BigInteger>>()
        {
            { typeof(sbyte),    value => new BigInteger((sbyte)value) },
            { typeof(byte),     value => new BigInteger((byte)value) },
            { typeof(short),    value => new BigInteger((short)value) },
            { typeof(ushort),   value => new BigInteger((ushort)value) },
            { typeof(int),      value => new BigInteger((int)value) },
            { typeof(uint),     value => new BigInteger((uint)value) },
            { typeof(long),     value => new BigInteger((long)value) },
            { typeof(ulong),    value => new BigInteger((ulong)value) },
        };

        public static BigInteger ToBigInteger(this object value)
        {
            var type = value.GetType();
            if (ToBigIntegerTable.ContainsKey(type))
            {
                return ToBigIntegerTable[type](value);
            }
            else
            {
                throw new InvalidCastException();
            }
        }
    }
}
