﻿using System.Collections.Generic;
using System.Diagnostics;
using System.Text.RegularExpressions;

namespace Nintendo.Log
{
    internal enum OpcodeType
    {
        None,
        GreaterThan,
        LessThan,
        GreaterEqual,
        LessEqual,
        Equal,
        NotEqual,
        Not,
        Match,
        NotMatch,
        And,
        Or
    }

    internal class FilterVariable
    {
        public FilterVariable(string name)
        {
            Name = name;
        }
        public string Name { get; set; }
    }

    internal class LogFilterParser
    {
        private static Dictionary<string, LogFilterParserNode> Cache = new Dictionary<string, LogFilterParserNode>();
        private static readonly int CacheCountMax = 10;

        private LogFilterLexer Lexer = new LogFilterLexer();
        private Dictionary<string, OpcodeType> OpcodeTable = new Dictionary<string, OpcodeType>()
        {
            { "<", OpcodeType.LessThan },
            { ">", OpcodeType.GreaterThan },
            { "<=", OpcodeType.LessEqual },
            { ">=", OpcodeType.GreaterEqual },
            { "==", OpcodeType.Equal },
            { "!=", OpcodeType.NotEqual },
            { "=~", OpcodeType.Match },
            { "!~", OpcodeType.NotMatch }
        };

        public LogFilterParserNode Evaluate(string text)
        {
            lock (Cache)
            {
                if (Cache.ContainsKey(text))
                {
                    return Cache[text];
                }
            }

            Lexer.Evaluate(text);
            var result = EvaluateOr();
            var token = Lexer.Peek();
            if (token != null)
            {
                if (token.TokenType == TokenType.LeftParen)
                {
                    throw new FilterSyntaxErrorException("Missing right parenthesis.");
                }
                else if (token.TokenType == TokenType.RightParen)
                {
                    throw new FilterSyntaxErrorException("Missing left parenthesis.");
                }
                else
                {
                    throw new FilterSyntaxErrorException(string.Empty);
                }
            }

            lock (Cache)
            {
                Debug.Assert(Cache.Count <= CacheCountMax);
                if (Cache.Count == CacheCountMax)
                {
                    Cache.Clear();
                }
                Cache[text] = result;
            }
            return result;
        }

        private LogFilterParserNode EvaluateOr()
        {
            var lhs = EvaluateAnd();
            while (true)
            {
                var token = Lexer.Peek();
                if (token != null && token.TokenType == TokenType.OrOperator)
                {
                    Lexer.Read();
                    var rhs = EvaluateAnd();
                    lhs = new LogFilterParserNode(lhs, rhs, OpcodeType.Or);
                }
                else
                {
                    break;
                }
            }
            return lhs;
        }

        private LogFilterParserNode EvaluateAnd()
        {
            var lhs = EvaluateComparison();
            while (true)
            {
                var token = Lexer.Peek();
                if (token != null && token.TokenType == TokenType.AndOperator)
                {
                    Lexer.Read();
                    var rhs = EvaluateComparison();
                    lhs = new LogFilterParserNode(lhs, rhs, OpcodeType.And);
                }
                else
                {
                    break;
                }
            }
            return lhs;
        }

        private LogFilterParserNode EvaluateComparison()
        {
            var lhs = EvaluateNot();
            while (true)
            {
                var token = Lexer.Peek();
                if (token != null && token.TokenType == TokenType.ComparisonOperator)
                {
                    Lexer.Read();
                    var rhs = EvaluateNot();
                    lhs = new LogFilterParserNode(lhs, rhs, OpcodeTable[token.Value]);
                }
                else
                {
                    break;
                }
            }
            return lhs;
        }

        private LogFilterParserNode EvaluateNot()
        {
            var token = Lexer.Peek();
            if (token == null)
            {
                throw new FilterSyntaxErrorException("Missing operand(s).");
            }
            else if (token.TokenType == TokenType.NotOperator)
            {
                Lexer.Read();
                var rhs = EvaluateNot();
                return new LogFilterParserNode(null, rhs, OpcodeType.Not);
            }
            else
            {
                return EvaluatePrimitive();
            }
        }

        private LogFilterParserNode EvaluatePrimitive()
        {
            var token = Lexer.Read();
            if (token == null)
            {
                throw new FilterSyntaxErrorException(string.Empty);
            }
            else if (token.TokenType == TokenType.LeftParen)
            {
                var result = EvaluateOr();
                token = Lexer.Read();
                if (token == null)
                {
                    throw new FilterSyntaxErrorException("Missing right parenthesis.");
                }
                else if (token.TokenType == TokenType.RightParen)
                {
                    return result;
                }
                else
                {
                    throw new FilterSyntaxErrorException(string.Empty);
                }
            }
            else if (token.TokenType == TokenType.Number)
            {
                var rhs = int.Parse(token.Value);
                return new LogFilterParserNode(null, rhs, OpcodeType.None);
            }
            else if (token.TokenType == TokenType.String)
            {
                var rhs = token.Value.Trim('"');
                return new LogFilterParserNode(null, rhs, OpcodeType.None);
            }
            else if (token.TokenType == TokenType.Regexp)
            {
                try
                {
                    var rhs = new Regex(token.Value.Trim('/'));
                    return new LogFilterParserNode(null, rhs, OpcodeType.None);
                }
                catch (System.ArgumentException)
                {
                    throw new FilterSyntaxErrorException("Invalid regexp pattern.");
                }
            }
            else if (token.TokenType == TokenType.Identifier)
            {
                var rhs = new FilterVariable(token.Value);
                return new LogFilterParserNode(null, rhs, OpcodeType.None);
            }
            else
            {
                throw new FilterSyntaxErrorException("Missing operand(s).");
            }
        }
    }
}
