﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text.RegularExpressions;

namespace Nintendo.Log
{
    internal enum TokenType
    {
        LeftParen,
        RightParen,
        Identifier,
        OrOperator,
        AndOperator,
        ComparisonOperator,
        NotOperator,
        String,
        Regexp,
        Number
    }

    internal class FilterToken
    {
        public FilterToken(TokenType type, string value)
        {
            TokenType = type;
            Value = value;
        }

        public override string ToString()
        {
            return
                "{" + Enum.GetName(typeof(TokenType), TokenType) + ",\"" + Value + "\"}";
        }

        public TokenType TokenType { get; set; }
        public string Value { get; set; }
    }

    internal class LogFilterLexer
    {
        private static readonly List<KeyValuePair<TokenType, string>> PatternMap =
            new List<KeyValuePair<TokenType, string>>()
        {
            { TokenType.LeftParen, "\\(" },
            { TokenType.RightParen, "\\)" },
            { TokenType.Identifier, "[_a-zA-Z][_a-zA-Z\\d]*" },
            { TokenType.OrOperator, "\\|\\|" },
            { TokenType.AndOperator, "&&" },
            { TokenType.ComparisonOperator, "<=|>=|<|>|==|!=|=~|!~" },
            { TokenType.NotOperator, "!" },
            { TokenType.String, "\"(\\\"|\\\\|\\n|[^\"])*?\"" }, // \\n は不要？
            { TokenType.Regexp, "/(\\/|\\\\|[^/])*?/" },
            { TokenType.Number, "\\d+" }
        };
        private static readonly Regex SpacePattern = new Regex("^\\s+");
        private static readonly Regex TokenPattern =
            new Regex(
                "^(" + string.Join("|",
                    (from pair in PatternMap
                     select "(?<" + Enum.GetName(typeof(TokenType), pair.Key) + ">" + pair.Value + ")"))
                + ")");
        private int CurrentIndex;
        private List<FilterToken> TokenList = new List<FilterToken>();

        public FilterToken Read()
        {
            var token = Peek();
            if (token != null)
            {
                CurrentIndex++;
            }
            return token;
        }

        public FilterToken Peek()
        {
            if (CurrentIndex < TokenList.Count)
            {
                return TokenList[CurrentIndex];
            }
            else
            {
                return null;
            }
        }

        private void SkipSpace(string text, ref int startIndex)
        {
            var match = SpacePattern.Match(text.Substring(startIndex));
            if (match.Success)
            {
                Debug.Assert(match.Index == 0);
                startIndex += match.Length;
            }
        }

        public void Evaluate(string text)
        {
            CurrentIndex = 0;
            TokenList.Clear();
            var index = 0;
            while (true)
            {
                SkipSpace(text, ref index);
                if (index == text.Length)
                {
                    break;
                }
                Debug.Assert(index < text.Length);
                var match = TokenPattern.Match(text.Substring(index));
                if (match.Success)
                {
                    FilterToken token = null;
                    foreach (TokenType type in Enum.GetValues(typeof(TokenType)))
                    {
                        var group = match.Groups[Enum.GetName(typeof(TokenType), type)];
                        if (!string.IsNullOrEmpty(group.Value))
                        {
                            Debug.Assert(group.Index == 0);
                            token = new FilterToken(
                                type, text.Substring(index, group.Length));
                            break;
                        }
                    }
                    Debug.Assert(token != null);
                    TokenList.Add(token);
                    index += token.Value.Length;
                }
                else
                {
                    throw new FilterSyntaxErrorException(text.Substring(index) + Environment.NewLine + "^ An invalid token is here.");
                }
            }
        }
    }

    static class ListExtensions
    {
        public static void Add(this List<KeyValuePair<TokenType, string>> list, TokenType type, string pattern)
            => list.Add(new KeyValuePair<TokenType, string>(type, pattern));
    }
}
