﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Xml.Linq;
using System.Xml.Serialization;

namespace Nintendo.Htcs
{
    public class HtcsCommunicator : IDisposable
    {
        private const int CHostDaemonPort = 8002;
        public TargetList TargetList { get; private set; }
        public PortMap PortMap { get; private set; }
        private TcpClient client;
        private Thread receiveThread;
        private bool disposed;
        private Queue<HtcDaemonAckEvent> AckQueue = new Queue<HtcDaemonAckEvent>();
        public event EventHandler<HtcsInfoUpdatedEventArgs> HtcsInfoUpdated;
        private event EventHandler<NewPortEventArgs> NewPortRegistered; // TODO 消す

        public HtcsCommunicator()
        {
            this.TargetList = new TargetList();
            this.PortMap = new PortMap();
            this.client = new TcpClient();
            this.receiveThread = new Thread(ReceiveFunc);
        }

        ~HtcsCommunicator()
        {
            this.Dispose(false);
        }

        public bool Start()
        {
            try
            {
                this.client.Connect("localhost", CHostDaemonPort);
                Logger.WriteLine("Connected to HtcDaemon with {0}.", client.Client.LocalEndPoint);
                this.receiveThread.Start();
            }
            catch (Exception e)
            {
                Logger.WriteLine(e);
                return false;
            }
            return true;
        }

        public void Stop()
        {
            this.Dispose();
        }

        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    this.client.Close();

                    try
                    {
                        this.receiveThread.Join();
                    }
                    catch (ThreadStateException)
                    {
                        // スレッド開始前にクラスが Dispose されたとしても、問題ではない
                    }
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }
        }

        public uint RegisterPort(PortMapItem portMapItem)
        {
            using (HtcDaemonAckEvent waitAckEvent = new HtcDaemonAckEvent())
            {
                lock (AckQueue)
                {
                    using (var sw = new StreamWriter(client.GetStream(), new UTF8Encoding(false), 1024, true))
                    {
                        var command = new RegisterPortMapCommand(portMapItem, "Auto");
                        string xml = command.ToXElement().ToString(SaveOptions.DisableFormatting);
                        sw.WriteLine(xml);
                    }
                    AckQueue.Enqueue(waitAckEvent);
                }
                waitAckEvent.Wait();
                return waitAckEvent.GetResult().GetValue();
            }
        }

        public uint RegisterPort(HtcsPeerName htcsPeerName, string htcsPortName, System.Net.IPEndPoint endPoint)
        {
            HtcsPortDescriptor htcsPortDesc = new HtcsPortDescriptor(htcsPeerName, htcsPortName);
            PortMapItem portMapping = new PortMapItem(htcsPortDesc, endPoint);
            return RegisterPort(portMapping);
        }

        public uint UnregisterPort(HtcsPortDescriptor htcsPortDescriptor)
        {
            using (HtcDaemonAckEvent waitAckEvent = new HtcDaemonAckEvent())
            {
                lock (AckQueue)
                {
                    using (var sw = new StreamWriter(client.GetStream(), new UTF8Encoding(false), 1024, true))
                    {
                        var command = new UnregisterPortMapCommand(htcsPortDescriptor, "Auto");
                        string xml = command.ToXElement().ToString(SaveOptions.DisableFormatting);
                        sw.WriteLine(xml);
                    }
                    AckQueue.Enqueue(waitAckEvent);
                }
                waitAckEvent.Wait();
                return waitAckEvent.GetResult().GetValue();
            }
        }

        public uint UnregisterPort(HtcsPeerName htcsPeerName, string htcsPortName)
        {
            HtcsPortDescriptor htcsPortDesc = new HtcsPortDescriptor(htcsPeerName, htcsPortName);
            return UnregisterPort(htcsPortDesc);
        }

        private void ReceiveFunc()
        {
            var sr = new StreamReader(client.GetStream(), new UTF8Encoding(false), false, 1024, true);

            while (true)
            {
                if (!ReceiveLine(sr))
                {
                    break;
                }
            }
        }

        private bool ReceiveLine(StreamReader sr)
        {
            try
            {
                var line = sr.ReadLine();
                if (line == null)
                {
                    return false;
                }
                var element = XElement.Parse(line);
                if (element.Name == HtcsInfo.XElementName)
                {
                    return ReceivePortMap(element);
                }
                else if (element.Name == CommandResult.XElementName)
                {
                    return ReceiveAck(element);
                }
            }
            catch (EndOfStreamException)
            {
                Logger.WriteLine("Disconnected HtcDaemon.");
                return false;
            }
            catch (IOException)
            {
                Logger.WriteLine("Disconnected HtcDaemon.");
                return false;
            }

            return true;
        }

        private bool ReceivePortMap(XElement element)
        {
            var htcsInfo = new HtcsInfo(element);
            var newTargets = htcsInfo.TargetList;
            var newMap = htcsInfo.PortMap;

            List<PortMapItem> newlyRegisteredItems = newMap.Except(this.PortMap).ToList();
            foreach (PortMapItem newItem in newlyRegisteredItems)
            {
                RaiseNewPort(newItem);
            }
            this.TargetList = newTargets;
            this.PortMap = newMap;

            if (HtcsInfoUpdated != null)
            {
                HtcsInfoUpdated(this, new HtcsInfoUpdatedEventArgs(this.TargetList, this.PortMap));
            }
            return true;
        }

        private bool ReceiveAck(XElement element)
        {
            HtcDaemonAckEvent signal;
            lock (AckQueue)
            {
                signal = AckQueue.Dequeue();
            }
            signal.SetResult(new CommandResult(element));
            signal.Signal();
            return true;
        }

        private void RaiseNewPort(PortMapItem portMapItem)
        {
            if (NewPortRegistered != null)
            {
                NewPortRegistered(this, new NewPortEventArgs(portMapItem));
            }
        }

        private class NewPortEventArgs : EventArgs
        {
            public NewPortEventArgs(PortMapItem portMapItem)
            {
                this.PortMapItem = portMapItem;
            }

            public PortMapItem PortMapItem { get; private set; }
        }
    }

    internal class HtcDaemonAckEvent : IDisposable
    {
        private bool disposed;
        private AutoResetEvent signal = new AutoResetEvent(false);
        private CommandResult result;

        ~HtcDaemonAckEvent()
        {
            this.Dispose(false);
        }
        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }
        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    signal.Dispose();
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }
        }
        public void Wait()
        {
            signal.WaitOne();
        }
        public void Signal()
        {
            signal.Set();
        }
        public void SetResult(CommandResult res)
        {
            result = res;
        }
        public CommandResult GetResult()
        {
            return result;
        }
    }

    public class HtcsInfoUpdatedEventArgs : EventArgs
    {
        public TargetList TargetList { get; private set; }
        public PortMap PortMap { get; private set; }

        public HtcsInfoUpdatedEventArgs(TargetList targetList, PortMap portMap)
        {
            this.TargetList = targetList;
            this.PortMap = portMap;
        }
    }
}
