﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Htcs;

namespace HtcDaemon.SerialConnection
{
    internal class SerialConnectionChannel : IChannel
    {
        private bool disposed;
        private SerialConnectionMux mux;
        private int channelId;
        private CancellationTokenSource cancellationTokenSource = new CancellationTokenSource();
        private ConcurrentQueue<Message> receiveQueue = new ConcurrentQueue<Message>();
        private AutoResetEvent receiveQueueEvent = new AutoResetEvent(false);
        private FlowController flowController;
        private ByteBlockQueue sendQueue = new ByteBlockQueue();

        public int ChannelId { get { return channelId; } }
        public HtcsPeerName TargetPeerName { get { return mux.TargetPeerName; } }

        public SerialConnectionChannel(SerialConnectionMux mux, int channelId, uint remoteBufferLength)
        {
            this.mux = mux;
            this.channelId = channelId;
            this.flowController = new FlowController(remoteBufferLength);
        }

        ~SerialConnectionChannel()
        {
            this.Dispose(false);
        }

        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        // ReadMessage が進行中の場合、完了してから Dispose すること
        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    this.cancellationTokenSource.Cancel();
                    this.receiveQueueEvent.Set();
                    this.receiveQueueEvent.Dispose();
                    this.cancellationTokenSource.Dispose();
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }
        }

        public void OnReceived(Message message)
        {
            receiveQueue.Enqueue(message);
            receiveQueueEvent.Set();
        }

        #region IChannel メンバー

        public Message ReceiveMessage()
        {
            var token = cancellationTokenSource.Token;

            Message message;
            while (!receiveQueue.TryDequeue(out message))
            {
                receiveQueueEvent.WaitOne();
                if (token.IsCancellationRequested)
                {
                    throw new ChannelDisconnectedException();
                }
            }
            return message;
        }

        public void SendMessage(Message message)
        {
            mux.SendMessage(message);
        }

        public void SendData(byte[] data, int offset, int count)
        {
            sendQueue.Enqueue(data, offset, count);
            mux.EnqueueSendRequest(this);
        }

        public void Cancel()
        {
            cancellationTokenSource.Cancel();
            receiveQueueEvent.Set();
        }

        #endregion

        public void SetAckedSeq(uint ackedSeq)
        {
            flowController.SetAckedSeq(ackedSeq);
        }

        public void AddSentSeq(uint dataLength)
        {
            flowController.AddSentSeq(dataLength);
        }

        public uint GetSendableLength()
        {
            return flowController.GetSendableLength();
        }

        public int GetSendQueueQueuedLength()
        {
            return sendQueue.Length;
        }

        public int DequeueFromSendQueue(byte[] buffer, int offset, int count)
        {
            return sendQueue.Dequeue(buffer, offset, count);
        }
    }
}
