﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.ConstrainedExecution;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;
using Microsoft.Win32.SafeHandles;

namespace HtcDaemon.Native
{
    internal static partial class NativeMethods
    {
        [StructLayout(LayoutKind.Explicit)]
        internal struct SP_DEVICE_INTERFACE_DATA
        {
            [FieldOffset(0)]
            public int cbSize;

            [FieldOffset(4)]
            public Guid Guid;

            [FieldOffset(20)]
            public uint Flags;

            [FieldOffset(24)]
            public IntPtr Reserved;
        }

        public const uint DIGCF_DEFAULT = 0x00000001; // only valid with DIGCF_DEVICEINTERFACE
        public const uint DIGCF_PRESENT = 0x00000002;
        public const uint DIGCF_ALLCLASSES = 0x00000004;
        public const uint DIGCF_PROFILE = 0x00000008;
        public const uint DIGCF_DEVICEINTERFACE = 0x00000010;

        [DllImport("setupapi.dll", SetLastError = true, CharSet = CharSet.Auto)]
        public static extern SafeDeviceInfoSetHandle SetupDiGetClassDevs(
                ref Guid ClassGuid,
                IntPtr NullEnumerator,
                IntPtr hwndParent,
                uint Flags);

        [DllImport("setupapi.dll", SetLastError = true)]
        [return: MarshalAs(UnmanagedType.Bool)]
        public static extern bool SetupDiDestroyDeviceInfoList(IntPtr DeviceInfoSet);

        [DllImport("setupapi.dll", SetLastError = true)]
        [return: MarshalAs(UnmanagedType.Bool)]
        public static extern bool SetupDiEnumDeviceInterfaces(
            SafeDeviceInfoSetHandle DeviceInfoSet,
            IntPtr NullDeviceInfoData,
            ref Guid InterfaceClassGuid,
            int MemberIndex,
            ref SP_DEVICE_INTERFACE_DATA DeviceInterfaceData);

        [DllImport("setupapi.dll", SetLastError = true, CharSet = CharSet.Auto)]
        [return: MarshalAs(UnmanagedType.Bool)]
        public static extern bool SetupDiGetDeviceInterfaceDetail(
            SafeDeviceInfoSetHandle DeviceInfoSet,
            ref SP_DEVICE_INTERFACE_DATA DeviceInterfaceData,
            IntPtr DeviceInterfaceDetailData,
            int DeviceInterfaceDetailDataSize,
            out int RequiredSize,
            IntPtr NullDeviceInfoData);

        public static bool SetupDiGetDeviceInterfaceDetail(
            SafeDeviceInfoSetHandle DeviceInfoSet,
            ref SP_DEVICE_INTERFACE_DATA DeviceInterfaceData,
            out string devicePath)
        {
            int requiredSize;
            SetupDiGetDeviceInterfaceDetail(DeviceInfoSet, ref DeviceInterfaceData, IntPtr.Zero, 0, out requiredSize, IntPtr.Zero);

            using (var detailData = new ScopedHGlobal(requiredSize))
            {
                // 32-bit版は、構造体が pack(1) になっている
                // DeviceInterfaceDetailData の構造
                //    uint   cbSize;            // 6 (32-bit)、8 (64-bit)
                //    char   DevicePath[1];
                // cbSize を書く
                Marshal.WriteInt32(detailData, 0, Environment.Is64BitProcess ? 8 : 6);

                if (!SetupDiGetDeviceInterfaceDetail(DeviceInfoSet, ref DeviceInterfaceData, detailData, requiredSize, out requiredSize, IntPtr.Zero))
                {
                    devicePath = string.Empty;
                    return false;
                }

                // DevicePath を読み出す。最後の NUL 文字を除いておく。
                devicePath = Marshal.PtrToStringAuto((IntPtr)detailData + 4, (requiredSize - 4) / sizeof(char) - 1);
                return true;
            }
        }
    }
}
