﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Htcs;

namespace HtcDaemon
{
    internal abstract class Mux : IDisposable
    {
        private IChannel[] channels
            = new IChannel[HtcConstants.HtcChannelCountMax];

        public abstract HtcsPeerName TargetPeerName { get; }
        public abstract string TargetEndPointString { get; }
        public abstract IChannel ControlChannel { get; }

        public abstract void Start();
        public abstract void Stop();
        protected abstract IChannel CreateDataChannelCore(int channelId, uint remoteBufferLength);

        private bool disposed;

        // 下位接続が切断された。この切断を起点に、あらゆるクリーンアップを行う。
        public event EventHandler<TargetEventArgs> Disconnected;

        ~Mux()
        {
            Dispose(false);
        }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    foreach (var channel in channels)
                    {
                        if (channel != null)
                        {
                            channel.Cancel();
                        }
                    }
                    foreach (var channel in channels)
                    {
                        if (channel != null)
                        {
                            channel.Dispose();
                        }
                    }
                }
                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }

            //// 基底クラスのDisposeを呼び出す
            // base.Dispose(disposing);
        }

        public IChannel CreateDataChannel(int channelId, uint remoteBufferLength)
        {
            var channel = CreateDataChannelCore(channelId, remoteBufferLength);
            lock (channels)
            {
                channels[channelId] = channel;
            }
            return channel;
        }

        public void DestroyDataChannel(int channelId)
        {
            IChannel channel;
            lock (channels)
            {
                channel = channels[channelId];
                channels[channelId] = null;
            }
            channel.Cancel();
            channel.Dispose();
        }

        protected void CancelAllChannels()
        {
            ControlChannel.Cancel();
            lock (channels)
            {
                foreach (var channel in channels.Where(c => c != null))
                {
                    channel.Cancel();
                }
            }
        }

        // TODO: 非同期に削除されるかも
        public IChannel GetChannel(int channelId)
        {
            if (channelId == HtcConstants.ControlChannelId)
            {
                return ControlChannel;
            }
            else
            {
                lock (channels)
                {
                    return channels[channelId];
                }
            }
        }

        protected void RaiseDisconnected()
        {
            if (Disconnected != null)
            {
                Disconnected(this, new TargetEventArgs(TargetPeerName));
            }
        }
    }
}
