﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Xml.Linq;
using HtcDaemon.DirectSocket;
using HtcDaemon.SerialConnection.SerialSocket;
using HtcDaemon.Usb;
using Nintendo.Htcs;

namespace HtcDaemon
{
    internal class HostDaemon : IDisposable
    {
        private bool disposed;
        private List<ITargetDetector> targetDetectors = new List<ITargetDetector>();
        private TcpServer hostAppListener = new TcpServer(HtcConstants.HostAppTcpPort);
        private List<HostTargetConnection> targetConnections = new List<HostTargetConnection>();
        private List<HostAppConnection> hostAppConnections = new List<HostAppConnection>();
        private HtcsInfo htcsInfo = new HtcsInfo();
        private PortMap hostPorts = new PortMap();
        private SerializedExecutor serializedExecutor = new SerializedExecutor();

        public HostDaemon()
        {
            if (ProgramOptions.DirectSocketEnabled)
            {
                var targetDetector = new DirectSocketTargetDetector(HtcConstants.TargetListenUdpPort);
                targetDetector.Detected += OnTargetDetected;
                targetDetectors.Add(targetDetector);
            }
            if (ProgramOptions.SerialSocketEnabled)
            {
                var targetDetector = new SerialSocketTargetDetector(HtcConstants.TargetListenUdpPortSerialConnection);
                targetDetector.Detected += OnTargetDetected;
                targetDetectors.Add(targetDetector);
            }
            if (ProgramOptions.UsbEnabled)
            {
                var targetDetector = new UsbTargetDetector();
                targetDetector.Detected += OnTargetDetected;
                targetDetectors.Add(targetDetector);
            }
            hostAppListener.Connected += OnHostAppConnected;
        }

        ~HostDaemon()
        {
            this.Dispose(false);
        }

        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    hostAppListener.Dispose();
                    serializedExecutor.ExecuteAsync(
                        () => DisconnectAll());
                    serializedExecutor.Dispose();
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }
        }

        public void Start()
        {
            serializedExecutor.Start();
            foreach (var targetDetector in targetDetectors)
            {
                targetDetector.Start();
            }
            hostAppListener.Start();
        }

        public void Stop()
        {
            Dispose();
        }

        private void DisconnectAll()
        {
            foreach (var targetDetector in targetDetectors)
            {
                targetDetector.Stop();
            }
            hostAppListener.Stop();
            foreach (var connection in targetConnections)
            {
                connection.Stop();
            }
        }

        public void DebugConsole()
        {
            var debugCommandExecutor = new DebugCommandExecutor(this);
            while (true)
            {
                string input = Console.ReadLine();
                debugCommandExecutor.Execute(input);
            }
        }

        private void OnTargetDetected(object sender, TargetDetectedEventArgs e)
        {
            serializedExecutor.ExecuteAsync(
                () =>
                {
                    var newConnection = e.Connection;

                    // 同じターゲットから接続された場合、
                    // 強制シャットダウン等で接続が残ったものとみなして既存の接続を切断する
                    var duplicatedConnections = new List<HostTargetConnection>();
                    lock (targetConnections)
                    {
                        duplicatedConnections.AddRange(targetConnections.Where(c => IsSameTarget(newConnection, c)));
                        targetConnections.RemoveAll(c => IsSameTarget(newConnection, c));

                        targetConnections.Add(newConnection);
                    }

                    foreach (var duplicatedConnection in duplicatedConnections)
                    {
                        duplicatedConnection.Stop();
                    }

                    Console.WriteLine("Connected with target from {0}", newConnection.TargetEndPointString);
                    newConnection.Disconnected += OnTargetDisconnected;
                    newConnection.TargetPortRegistered += OnTargetPortRegistered;
                    newConnection.TargetPortUnregistered += OnTargetPortUnregistered;
                    newConnection.Start();
                    lock (hostPorts)
                    {
                        foreach (var hostPort in hostPorts)
                        {
                            newConnection.NotifyHostPortRegistered(hostPort);
                        }
                    }
                });
        }

        // TODO: 同じターゲットであると判定する方法
        private static bool IsSameTarget(HostTargetConnection newConnection, HostTargetConnection c)
        {
            return c.TargetPeerName == newConnection.TargetPeerName;
        }

        // 切断されたので、接続をクリーンアップする
        private void OnTargetDisconnected(object sender, TargetEventArgs e)
        {
            // 接続の管理は、HostDaemon のスレッドで行い、イベントで同期的に削除しないようにする
            serializedExecutor.ExecuteAsync(
                () =>
                {
                    var targetConnection = (HostTargetConnection)sender;
                    lock (targetConnections)
                    {
                        targetConnections.Remove(targetConnection);
                    }

                    targetConnection.Dispose();
                });
        }

        private void OnHostAppConnected(object sender, TcpConnectedEventArgs e)
        {
            serializedExecutor.ExecuteAsync(
                () =>
                {
                    var connection = MakeHostConnection(e.Socket);
                    lock (hostAppConnections)
                    {
                        hostAppConnections.Add(connection);
                    }
                    connection.Start();

                    lock (htcsInfo)
                    {
                        var xml = GetHtcsInfoXml();
                        connection.Distribute(xml);
                    }
                });
        }

        private HostAppConnection MakeHostConnection(Socket socket)
        {
            Console.WriteLine("Connected with HostApp from {0}", socket.RemoteEndPoint);

            var connection = new HostAppConnection(socket);
            connection.Disconnected += OnHostAppDisconnected;
            connection.HostPortRegistered += OnHostPortRegistered;
            connection.HostPortUnregistered += OnHostPortUnregistered;

            return connection;
        }

        private void OnHostAppDisconnected(object sender, EventArgs e)
        {
            serializedExecutor.ExecuteAsync(
                () =>
                {
                    var connection = (HostAppConnection)sender;
                    Console.WriteLine("Disconnected from HostApp");
                    lock (hostAppConnections)
                    {
                        hostAppConnections.Remove(connection);
                    }
                    connection.Dispose();
                });
        }

        private void OnTargetPortRegistered(object sender, TargetPortRegisteredEventArgs e)
        {
            serializedExecutor.ExecuteAsync(
                () =>
                {
                    htcsInfo.TargetList.Add(e.Target);
                    htcsInfo.PortMap.Add(e.PortMapItem);
                    NotifyTargetPortUpdate();
                });
        }

        private void OnTargetPortUnregistered(object sender, TargetPortUnregisteredEventArgs e)
        {
            serializedExecutor.ExecuteAsync(
                () =>
                {
                    var target = htcsInfo.TargetList.First(x => x.HtcsPeerName == e.HtcsPortDescriptor.HtcsPeerName.Value);
                    htcsInfo.TargetList.Remove(target);
                    htcsInfo.PortMap.Remove(e.HtcsPortDescriptor);
                    NotifyTargetPortUpdate();
                });
        }

        private void NotifyTargetPortUpdate()
        {
            var xml = GetHtcsInfoXml();
            DistributeToAll(xml);
        }

        private void DistributeToAll(string xml)
        {
            lock (hostAppConnections)
            {
                foreach (var connection in hostAppConnections)
                {
                    connection.Distribute(xml);
                }
            }
        }

        private string GetHtcsInfoXml()
        {
            return htcsInfo.ToXElement().ToString(SaveOptions.DisableFormatting);
        }

        private void OnHostPortRegistered(object sender, HostPortRegisteredEventArgs e)
        {
            serializedExecutor.ExecuteAsync(
                () =>
                {
                    lock (hostPorts)
                    {
                        hostPorts.Add(e.PortMapItem);
                    }

                    uint resultValue = 0;

                    foreach (var connection in targetConnections)
                    {
                        connection.NotifyHostPortRegistered(e.PortMapItem);
                    }

                    CommandResult result = new CommandResult(e.RequestName, resultValue);
                    ((HostAppConnection)sender).Distribute(result.ToXElement().ToString());
                });
        }

        private void OnHostPortUnregistered(object sender, HostPortUnregisteredEventArgs e)
        {
            serializedExecutor.ExecuteAsync(
                () =>
                {
                    lock (hostPorts)
                    {
                        hostPorts.Remove(e.HtcsPortDescriptor);
                    }

                    uint resultValue = 0;

                    foreach (var connection in targetConnections)
                    {
                        connection.NotifyHostPortUnregistered(e.HtcsPortDescriptor);
                    }

                    CommandResult result = new CommandResult(e.RequestName, resultValue);
                    ((HostAppConnection)sender).Distribute(result.ToXElement().ToString());
                });
        }

        private void DumpPortMap()
        {
            Console.WriteLine("---Connections with Targets---");
            foreach (var targetConnection in targetConnections)
            {
                Console.WriteLine("{0}, {1}", targetConnection.TargetPeerName, targetConnection.TargetEndPointString);
            }
            Console.WriteLine("---PortMap on Host---");
            foreach (var hostPort in hostPorts)
            {
                Console.WriteLine(hostPort);
            }
            Console.WriteLine("---PortMap on Target---");
            foreach (var targetPort in htcsInfo.PortMap)
            {
                Console.WriteLine(targetPort);
            }
            Console.WriteLine("------");
        }

        private class DebugCommandExecutor
        {
            private static readonly char[] DebugCommandDelimiters = new char[] { ' ' };
            private HostDaemon parent;

            public DebugCommandExecutor(HostDaemon parentDaemon)
            {
                this.parent = parentDaemon;
            }

            public void Execute(string line)
            {
                if (string.IsNullOrEmpty(line))
                {
                    this.parent.DumpPortMap();
                }
                else
                {
                    try
                    {
                        var args = line.Split(DebugCommandDelimiters, StringSplitOptions.RemoveEmptyEntries);
                        switch (args[0])
                        {
                            case "":
                                this.parent.DumpPortMap();
                                break;
                            case "connect":
                                ConnectCommand(args);
                                break;
                            default:
                                Console.WriteLine("unknown command {0}", args[0]);
                                break;
                        }
                    }
                    catch (Exception e)
                    {
                        Console.WriteLine("{0}", e.Message);
                    }
                }
            }

            private void ConnectCommand(string[] args)
            {
                if (args.Length < 2)
                {
                    Console.WriteLine("connect ip_address [port]");
                    return;
                }

                var targetDetector = this.parent.targetDetectors.First() as DirectSocketTargetDetector;
                if (targetDetector == null)
                {
                    Console.WriteLine("Can only be used in the direct-socket mode (currently in the usb mode).");
                    return;
                }

                IPAddress address = ParseHelper(IPAddress.Parse, args[1], "ip_address");
                int port = args.Length >= 3 ? ParseHelper(int.Parse, args[2], "port") : HtcConstants.DefaultTargetTcpPort;

                targetDetector.MakeTargetConnectionAndRaiseDetected(new IPEndPoint(address, port));
            }

            private T ParseHelper<T>(Func<string, T> parseFunc, string s, string paramName)
            {
                try
                {
                    return parseFunc(s);
                }
                catch (Exception e)
                {
                    throw new ArgumentException(
                        string.Format(CultureInfo.CurrentCulture, "{0}: {1}", s, e.Message), paramName);
                }
            }
        }
    }
}
