﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Xml.Serialization;
using System.Net;
using System.Xml.Linq;
using Nintendo.Htcs;

namespace HtcDaemon
{
    internal class HostAppConnection : IDisposable
    {
        private CancellationTokenSource cancellationTokenSource = new CancellationTokenSource();
        private Thread communicationThread;
        private Socket socket;
        private NetworkStream stream;
        private bool disposed;
        private List<PortMapItem> portMapItems = new List<PortMapItem>();

        public event EventHandler Disconnected;
        public event EventHandler<HostPortRegisteredEventArgs> HostPortRegistered;
        public event EventHandler<HostPortUnregisteredEventArgs> HostPortUnregistered;

        public HostAppConnection(Socket socket)
        {
            this.socket = socket;
            this.stream = new NetworkStream(socket, false);
            this.communicationThread = new Thread(ComunicationThread);
        }

        ~HostAppConnection()
        {
            this.Dispose(false);
        }

        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    cancellationTokenSource.Cancel();
                    stream.Close();
                    try
                    {
                        socket.Shutdown(SocketShutdown.Both);
                    }
                    catch (SocketException)
                    {
                        // RST を受信していた場合、Shutdown から SocketException がスローされる
                    }
                    socket.Close();
                    communicationThread.Join();
                    cancellationTokenSource.Dispose();
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }
        }

        public void Distribute(string xml)
        {
            try
            {
                using (var writer = new StreamWriter(stream, new UTF8Encoding(false), 1024, true))
                {
                    // TORIAEZU: 改行をメッセージの終端とする
                    writer.WriteLine(xml);
                }
            }
            catch (EndOfStreamException)
            {
            }
            catch (IOException)
            {
            }
        }

        public void Start()
        {
            communicationThread.Start();
        }

        public void Stop()
        {
            Dispose();
        }

        public void ComunicationThread()
        {
            var token = cancellationTokenSource.Token;

            var sr = new StreamReader(stream, new UTF8Encoding(false), true, 1024, true);
            while (!token.IsCancellationRequested)
            {
                try
                {
                    var line = sr.ReadLine();
                    if (line == null)
                    {
                        break;
                    }

                    var element = XElement.Parse(line);
                    switch (element.Name.LocalName)
                    {
                        case RegisterPortMapCommand.XElementName:
                            ProcessRegisterPort(element);
                            break;
                        case UnregisterPortMapCommand.XElementName:
                            ProcessUnregisterPort(element);
                            break;
                    }
                }
                catch (Exception)
                {
                    break;
                }
            }

            try
            {
                OnDisconnect();
            }
            catch (ChannelDisconnectedException)
            {
            }
        }

        private void OnDisconnect()
        {
            foreach (var item in portMapItems)
            {
                RaiseHostPortUnregistered(item.HtcsPortDescriptor, string.Empty);
            }
            portMapItems.Clear();

            RaiseDisconnected();
        }

        private void ProcessRegisterPort(XElement element)
        {
            var command = new RegisterPortMapCommand(element);
            var portMapItem = FixIPAddressAny(command.PortMapItem);
            portMapItems.Add(portMapItem);
            RaiseHostPortRegistered(portMapItem, command.RequestName);
        }

        // IP アドレスが IPAddress.Any であった場合、対向のアドレスに補正する
        private PortMapItem FixIPAddressAny(PortMapItem portMapItem)
        {
            if (portMapItem.EndPoint.Address.Equals(IPAddress.Any))
            {
                var remoteAddress = ((IPEndPoint)socket.RemoteEndPoint).Address;
                return new PortMapItem(
                    portMapItem.HtcsPortDescriptor,
                    new IPEndPoint(remoteAddress, portMapItem.EndPoint.Port));
            }
            else
            {
                return portMapItem;
            }
        }

        private void ProcessUnregisterPort(XElement element)
        {
            var command = new UnregisterPortMapCommand(element);
            portMapItems.RemoveAt(portMapItems.FindIndex(d => d.HtcsPortDescriptor == command.HtcsPortDescriptor));
            RaiseHostPortUnregistered(command.HtcsPortDescriptor, command.RequestName);
        }

        private void RaiseDisconnected()
        {
            if (Disconnected != null)
            {
                Disconnected(this, new EventArgs());
            }
        }

        private void RaiseHostPortRegistered(PortMapItem portMapItem, string requestName)
        {
            if (HostPortRegistered != null)
            {
                HostPortRegistered(this, new HostPortRegisteredEventArgs(portMapItem, requestName));
            }
        }

        private void RaiseHostPortUnregistered(HtcsPortDescriptor htcsPortDescriptor, string requestName)
        {
            if (HostPortUnregistered != null)
            {
                HostPortUnregistered(this, new HostPortUnregisteredEventArgs(htcsPortDescriptor, requestName));
            }
        }
    }
}
