﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace HtcDaemon.DirectSocket
{
    internal class DirectSocketTargetDetector : ITargetDetector
    {
        public event EventHandler<TargetDetectedEventArgs> Detected;

        private bool disposed;
        private Thread listenThread;
        private CancellationTokenSource cancellationTokenSource = new CancellationTokenSource();
        private UdpClient client;

        // throws SocketException
        public DirectSocketTargetDetector(int udpPort)
        {
            listenThread = new Thread(ListenThread) { Name = "DirectSocketTargetDetector" };
            client = new UdpClient(new IPEndPoint(IPAddress.Any, udpPort));
        }

        ~DirectSocketTargetDetector()
        {
            this.Dispose(false);
        }

        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    cancellationTokenSource.Cancel();
                    client.Close();
                    listenThread.Join();
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }

            //// 基底クラスのDisposeを呼び出す
            // base.Dispose(disposing);
        }

        public void Start()
        {
            listenThread.Start();
        }

        public void Stop()
        {
            Dispose();
        }

        public void ListenThread()
        {
            var token = cancellationTokenSource.Token;
            while (!token.IsCancellationRequested)
            {
                var targetEP = new IPEndPoint(IPAddress.Any, 0);
                byte[] data;

                try
                {
                    data = client.Receive(ref targetEP);
                }
                catch (SocketException)
                {
                    // client が Close された
                    return;
                }

                if (data.Length < 4)
                {
                    continue;
                }

                Console.WriteLine("Got broadcast packet from {0}", targetEP.Address);
                int protocolVersion = BitConverter.ToUInt16(data, 0);
                int targetListeningPort = BitConverter.ToUInt16(data, 2);

                MakeTargetConnectionAndRaiseDetected(new IPEndPoint(targetEP.Address, targetListeningPort));
            }
        }

        // 検出されたターゲットに接続
        // デバッグコマンド用に公開
        public void MakeTargetConnectionAndRaiseDetected(IPEndPoint endPoint)
        {
            Socket socket;
            try
            {
                socket = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);
                socket.Connect(endPoint);
            }
            catch (SocketException ex)
            {
                Console.WriteLine(ex.Message);
                return;
            }

            HostTargetConnection connection;
            try
            {
#if SERIALCONNECTION
                connection = new HtcDaemon.SerialConnection.SocketStream.SocketStreamHostTargetConnectionBuilder().Build(socket);
#else
                connection = new DirectSocketHostTargetConnectionBuilder().Build(socket);
#endif
            }
            catch (IOException ex)
            {
                Console.WriteLine("{0} との接続の初期化に失敗しました。: {1}", socket.RemoteEndPoint, ex.Message);
                socket.Close();
                return;
            }

            RaiseDetected(connection);
        }

        private void RaiseDetected(HostTargetConnection connection)
        {
            if (Detected != null)
            {
                Detected(this, new TargetDetectedEventArgs(connection));
            }
        }
    }
}
