﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Htcs;

namespace HtcDaemon
{
    internal enum ControlMessageType : byte
    {
        // Target <-> Host
        RegisterPort,
        UnregisterPort,
        CloseChannel,

        // Target -> Host
        Connect,
        Accept,

        // Result
        ChannelResult
    }

    internal enum ControlMessageResultCode : byte
    {
        Success,
        PortNotFound,
        ConnectionFailed
    }

    internal abstract class ControlCommunicator : IDisposable
    {
        // TODO: 場所
        // ushort type;
        private const int ControlMessageExtraHeaderLength = 2;

        private bool disposed;
        private CancellationTokenSource cancellationTokenSource = new CancellationTokenSource();
        private Mux mux;
        private Thread receiveThread = null;
        private TargetPortManager targetPortManager = new TargetPortManager();
        private HostPortManager hostPortManager = new HostPortManager();
        private BridgeManager bridgeManager = new BridgeManager();

        public event EventHandler<TargetPortRegisteredEventArgs> TargetPortRegistered;
        public event EventHandler<TargetPortUnregisteredEventArgs> TargetPortUnregistered;

        protected abstract IHostAppListener CreateHostAppListener(string htcsPortName, int port);

        public ControlCommunicator(Mux mux)
        {
            this.receiveThread = new Thread(ReceiveThread) { Name = "ControlCommunicator.ReceiveThread" };
            this.mux = mux;

            this.bridgeManager.BridgeDisconnected += OnBridgeDisconnected;
            this.bridgeManager.HostAppDisconnected += OnHostAppDisconnected;
        }

        ~ControlCommunicator()
        {
            Dispose(false);
        }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    cancellationTokenSource.Cancel();
                    mux.ControlChannel.Cancel();
                    receiveThread.Join();
                    cancellationTokenSource.Dispose();
                    targetPortManager.Dispose();
                    bridgeManager.Dispose();
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }
        }

        public void Start()
        {
            receiveThread.Start();
        }

        public void Stop()
        {
            Dispose();
        }

        public void NotifyHostPortRegistered(PortMapItem portMapItem)
        {
            hostPortManager.RegisterHostPort(portMapItem.HtcsPortDescriptor.HtcsPortName, portMapItem.EndPoint);

            // TORIAEZU: べた書き
            var messageWriter = MakeControlMessageWriter(HtcConstants.HtcPortNameLength + 2, ControlMessageType.RegisterPort);
            messageWriter.Append(StringUtil.StringToCStyleString(portMapItem.HtcsPortDescriptor.HtcsPortName, HtcConstants.HtcPortNameLength));
            messageWriter.Append((ushort)portMapItem.EndPoint.Port);  // TORIAEZU: DirectSocket のためにポート番号を送信

            WriteMessage(messageWriter.MakeMessage());
        }

        public void NotifyHostPortUnregistered(HtcsPortDescriptor htcsPortDescriptor)
        {
            hostPortManager.UnregisterHostPort(htcsPortDescriptor.HtcsPortName);

            // TORIAEZU: べた書き
            var messageWriter = MakeControlMessageWriter(HtcConstants.HtcPortNameLength + 2, ControlMessageType.UnregisterPort);
            messageWriter.Append(StringUtil.StringToCStyleString(htcsPortDescriptor.HtcsPortName, HtcConstants.HtcPortNameLength));

            WriteMessage(messageWriter.MakeMessage());
        }

        public void RequestCloseChannel(int channelId)
        {
            // TORIAEZU: べた書き
            var messageWriter = MakeControlMessageWriter(2, ControlMessageType.CloseChannel);
            messageWriter.Append((ushort)channelId);

            WriteMessage(messageWriter.MakeMessage());
        }

        private void WriteChannelResultMessage(int channelId, ControlMessageResultCode resultCode)
        {
            // TORIAEZU: べた書き
            var messageWriter = MakeControlMessageWriter(2 + 2, ControlMessageType.ChannelResult);
            messageWriter.Append((ushort)channelId);
            messageWriter.Append((ushort)resultCode);

            WriteMessage(messageWriter.MakeMessage());
        }

        private void WriteMessage(Message message)
        {
            try
            {
                mux.ControlChannel.SendMessage(message);
            }
            catch (ChannelDisconnectedException)
            {
                // mux から Disconnected イベントが上がってくるので、そのタイミングで破棄してもらう
            }
        }

        private static MessageWriter MakeControlMessageWriter(int bodyLength, ControlMessageType type)
        {
            var messageWriter = new MessageWriter(HtcConstants.ControlChannelId, bodyLength + ControlMessageExtraHeaderLength);
            messageWriter.Append((ushort)type);
            return messageWriter;
        }

        private void OnReceived(Message message)
        {
            using (var r = message.MakeBodyReader())
            {
                ushort type = r.ReadUInt16();
                switch (type)
                {
                    case (byte)ControlMessageType.RegisterPort:
                        {
                            var portName = StringUtil.StringFromCStyleString(r.ReadBytes(HtcConstants.HtcPortNameLength));
                            int tcpPort = r.ReadUInt16();  // TORIAEZU: DirectSocket のためにポート番号を取得

                            var listener = CreateHostAppListener(portName, tcpPort);
                            var portMapItem = new PortMapItem(
                                new HtcsPortDescriptor(mux.TargetPeerName, portName), listener.LocalEndPoint);

                            targetPortManager.RegisterTargetPort(portName, listener);
                            RaiseTargetPortRegistered(portMapItem);
                            // TORIAEZU: 結果を通知しない
                            break;
                        }
                    case (byte)ControlMessageType.UnregisterPort:
                        {
                            var portName = StringUtil.StringFromCStyleString(r.ReadBytes(HtcConstants.HtcPortNameLength));

                            var htcsPortDescriptor = new HtcsPortDescriptor(mux.TargetPeerName, portName);

                            try
                            {
                                RaiseTargetPortUnregistered(htcsPortDescriptor);
                                targetPortManager.UnregisterTargetPort(portName);
                            }
                            catch (KeyNotFoundException)
                            {
                                // TORIAEZU: 結果を通知しない
                            }
                            break;
                        }
                    case (byte)ControlMessageType.Connect:
                        {
                            var portName = StringUtil.StringFromCStyleString(r.ReadBytes(HtcConstants.HtcPortNameLength));
                            var channelId = r.ReadUInt16();
                            var remoteBufferLength = r.ReadUInt32();

                            IPEndPoint endPoint;
                            try
                            {
                                endPoint = hostPortManager.GetEndPoint(portName);
                            }
                            catch (KeyNotFoundException)
                            {
                                WriteChannelResultMessage(channelId, ControlMessageResultCode.PortNotFound);
                                break;
                            }

                            var channel = mux.CreateDataChannel(channelId, remoteBufferLength);
                            Task.Run(() =>
                                {
                                    Socket socket;
                                    try
                                    {
                                        socket = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);
                                        socket.Connect(endPoint);
                                    }
                                    catch (SocketException)
                                    {
                                        WriteChannelResultMessage(channelId, ControlMessageResultCode.ConnectionFailed);
                                        return;
                                    }

                                    bridgeManager.CreateBridge(channel, socket);
                                    WriteChannelResultMessage(channelId, ControlMessageResultCode.Success);
                                });

                            break;
                        }
                    case (byte)ControlMessageType.Accept:
                        {
                            var portName = StringUtil.StringFromCStyleString(r.ReadBytes(HtcConstants.HtcPortNameLength));
                            var channelId = r.ReadUInt16();
                            var remoteBufferLength = r.ReadUInt32();

                            var channel = mux.CreateDataChannel(channelId, remoteBufferLength);

                            Task.Run(() =>
                                {
                                    // TORIAEZU: 別スレッドで同期 Accept を行う
                                    Socket socket;
                                    try
                                    {
                                        socket = targetPortManager.Accept(portName);
                                    }
                                    catch (KeyNotFoundException)
                                    {
                                        WriteChannelResultMessage(channelId, ControlMessageResultCode.PortNotFound);
                                        return;
                                    }
                                    catch (SocketException)
                                    {
                                        WriteChannelResultMessage(channelId, ControlMessageResultCode.ConnectionFailed);
                                        return;
                                    }

                                    // CreateBridge
                                    bridgeManager.CreateBridge(channel, socket);
                                    WriteChannelResultMessage(channelId, ControlMessageResultCode.Success);
                                });

                            break;
                        }
                    case (byte)ControlMessageType.CloseChannel:
                        {
                            var channelIdToClose = r.ReadUInt16();
                            bridgeManager.DestroyBridge(channelIdToClose);

                            break;
                        }
                }
            }
        }

        private void ReceiveThread()
        {
            var token = cancellationTokenSource.Token;
            while (!token.IsCancellationRequested)
            {
                try
                {
                    var message = mux.ControlChannel.ReceiveMessage();
                    OnReceived(message);
                }
                catch (ChannelDisconnectedException)
                {
                    // mux から Disconnected イベントが上がるので、それを起点として HostDaemon に破棄してもらう
                    break;
                }
            }
        }

        private void OnBridgeDisconnected(object sender, ChannelEventArgs e)
        {
            // TODO: もう mux が破棄されているかもしれない
            bridgeManager.DestroyBridge(e.Channel.ChannelId);
        }

        private void OnHostAppDisconnected(object sender, ChannelEventArgs e)
        {
            RequestCloseChannel(e.Channel.ChannelId);
        }

        private void RaiseTargetPortRegistered(PortMapItem portMapItem)
        {
            if (TargetPortRegistered != null)
            {
                TargetPortRegistered(this, new TargetPortRegisteredEventArgs(portMapItem));
            }
        }

        private void RaiseTargetPortUnregistered(HtcsPortDescriptor htcsPortDescriptor)
        {
            if (TargetPortUnregistered != null)
            {
                TargetPortUnregistered(this, new TargetPortUnregisteredEventArgs(htcsPortDescriptor));
            }
        }
    }
}
