﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.FsFileCacheSimulator.FileSystem
{
    internal class AccessTimeCalculator
    {
        // size バイト読み込みにかかる時間 (ナノ秒)
        // 出典: http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=185130689
        //       Aligned, RandomAccess の結果を採用する
        // TODO: StorageType だけでなく FileSystemType によっても変動し得るので、データを分けないといけない
        // TODO: StorageType, FileSystemType の充実 (← も ↑ もどこまで正確にシミュレーションするか次第)
        // TODO: 秒数でなく線形補間用の係数をテーブルで持つようにする
        private static readonly Dictionary<StorageType, (long size, long ns)[]> s_ReadTimeTable = new Dictionary<StorageType, (long, long)[]>()
        {
            { StorageType.NandFaster, new (long, long)[]
                {
                    (    1 * 1024,    303030),
                    (    2 * 1024,    425532),
                    (    4 * 1024,    500000),
                    (    8 * 1024,    567376),
                    (   16 * 1024,    661157),
                    (   32 * 1024,    774818),
                    (   64 * 1024,   1082910),
                    (  128 * 1024,   1732070),
                    (  256 * 1024,   3022432),
                    (  512 * 1024,   5644983),
                    ( 1024 * 1024,  10700104),
                    ( 2048 * 1024,  18823529),
                    ( 4096 * 1024,  32952534),
                }
            },
            { StorageType.NandSlower, new (long, long)[]
                {
                    (    1 * 1024,  10000000),
                    (    2 * 1024,  10000000),
                    (    4 * 1024,  10000000),
                    (    8 * 1024,  11428571),
                    (   16 * 1024,  11428571),
                    (   32 * 1024,  12307692),
                    (   64 * 1024,  13617021),
                    (  128 * 1024,  17297297),
                    (  256 * 1024,  23272727),
                    (  512 * 1024,  36571429),
                    ( 1024 * 1024,  81920000),
                    ( 2048 * 1024, 146285714),
                    ( 4096 * 1024, 273066667),
                }
            },
            { StorageType.Nand, new (long, long)[]
                {
                    (    1 * 1024,    476190),
                    (    2 * 1024,    645161),
                    (    4 * 1024,    740741),
                    (    8 * 1024,    824742),
                    (   16 * 1024,    930233),
                    (   32 * 1024,   1084746),
                    (   64 * 1024,   1495327),
                    (  128 * 1024,   2529644),
                    (  256 * 1024,   4406196),
                    (  512 * 1024,   7496340),
                    ( 1024 * 1024,  13044586),
                    ( 2048 * 1024,  22140541),
                    ( 4096 * 1024,  37304189),
                }
            },
            { StorageType.SdCard, new (long, long)[]
                {
                    (    1 * 1024,    833333),
                    (    2 * 1024,   1250000),
                    (    4 * 1024,   1428571),
                    (    8 * 1024,   1632653),
                    (   16 * 1024,   1702128),
                    (   32 * 1024,   2442748),
                    (   64 * 1024,   3786982),
                    (  128 * 1024,   6844920),
                    (  256 * 1024,  12736318),
                    (  512 * 1024,  24380952),
                    ( 1024 * 1024,  48530806),
                    ( 2048 * 1024,  97061611),
                    ( 4096 * 1024, 193207547),
                }
            },
            { StorageType.GameCard50MHz, new (long, long)[]
                {
                    (    1 * 1024,   1428571),
                    (    2 * 1024,   1666667),
                    (    4 * 1024,   1904762),
                    (    8 * 1024,   2105263),
                    (   16 * 1024,   2352941),
                    (   32 * 1024,   2782609),
                    (   64 * 1024,   3878788),
                    (  128 * 1024,   6530612),
                    (  256 * 1024,  10199203),
                    (  512 * 1024,  17655172),
                    ( 1024 * 1024,  34362416),
                    ( 2048 * 1024,  63405573),
                    ( 4096 * 1024, 118724638),
                }
            },
            { StorageType.GameCard25MHz, new (long, long)[]
                {
                    (    1 * 1024,   1666667),
                    (    2 * 1024,   2000000),
                    (    4 * 1024,   2352941),
                    (    8 * 1024,   2580645),
                    (   16 * 1024,   3076923),
                    (   32 * 1024,   3950617),
                    (   64 * 1024,   5765766),
                    (  128 * 1024,  10158730),
                    (  256 * 1024,  16953642),
                    (  512 * 1024,  30476190),
                    ( 1024 * 1024,  59883041),
                    ( 2048 * 1024, 113777778),
                    ( 4096 * 1024, 219037433),
                }
            }
        };
        private static readonly Dictionary<StorageType, (long size, long ns)[]> s_WriteTimeTable = new Dictionary<StorageType, (long, long)[]>()
        {
            { StorageType.NandSlower, new (long, long)[]
                {
                    (    1 * 1024,  10000000),
                    (    2 * 1024,  20000000),
                    (    4 * 1024,  20000000),
                    (    8 * 1024,  26666667),
                    (   16 * 1024,  13333333),
                    (   32 * 1024,   2689076),
                    (   64 * 1024,   4740741),
                    (  128 * 1024,   7950311),
                    (  256 * 1024,  13061224),
                    (  512 * 1024,  22164502),
                    ( 1024 * 1024,  43760684),
                    ( 2048 * 1024,  87148936),
                    ( 4096 * 1024, 171380753),
                }
            },
            { StorageType.Nand, new (long, long)[]
                {
                    (    1 * 1024,    909091),
                    (    2 * 1024,   1333333),
                    (    4 * 1024,   1538462),
                    (    8 * 1024,   1702128),
                    (   16 * 1024,   1333333),
                    (   32 * 1024,   1280000),
                    (   64 * 1024,   2012579),
                    (  128 * 1024,   3987539),
                    (  256 * 1024,   6109785),
                    (  512 * 1024,  10778947),
                    ( 1024 * 1024,  20480000),
                    ( 2048 * 1024,  39536680),
                    ( 4096 * 1024,  77870722),
                }
            },
            { StorageType.SdCard, new (long, long)[]
                {
                    (    1 * 1024,  10000000),
                    (    2 * 1024,  10000000),
                    (    4 * 1024,   8000000),
                    (    8 * 1024,  10000000),
                    (   16 * 1024,  10666667),
                    (   32 * 1024,  11034483),
                    (   64 * 1024,  20000000),
                    (  128 * 1024,  36571429),
                    (  256 * 1024,  56888889),
                    (  512 * 1024, 104489796),
                    ( 1024 * 1024, 200784314),
                    ( 2048 * 1024, 372363636),
                    ( 4096 * 1024, 585142857),
                }
            }
        };

        public static TimeSpan GetReadAccessTime(ulong size, FileSystemType fileSystemType, StorageType storageType)
        {
            if (!s_ReadTimeTable.ContainsKey(storageType))
            {
                throw new NotImplementedException();
            }
            var timeTable = s_ReadTimeTable[storageType];

            if (size > long.MaxValue)
            {
                throw new ArgumentException();
            }
            return CalculateAccessTime((long)size, timeTable);
        }

        public static TimeSpan GetWriteAccessTime(ulong size, FileSystemType fileSystemType, StorageType storageType)
        {
            if (!s_WriteTimeTable.ContainsKey(storageType))
            {
                throw new NotImplementedException();
            }
            var timeTable = s_WriteTimeTable[storageType];

            if (size > long.MaxValue)
            {
                throw new ArgumentException();
            }
            return CalculateAccessTime((long)size, timeTable);
        }

        private static TimeSpan CalculateAccessTime(long size, (long size, long ns)[] timeTable)
        {
            if (timeTable.Length < 2)
            {
                throw new ImplementationErrorException();
            }

            var startIndex = 0;
            var endIndex = timeTable.Length;
            while (endIndex - startIndex > 1)
            {
                var index = startIndex + (endIndex - startIndex) / 2;
                if (size < timeTable[index].size)
                {
                    endIndex = index;
                }
                else
                {
                    startIndex = index;
                }
            }
            if (endIndex == timeTable.Length)
            {
                endIndex--;
                startIndex--;
            }

            // 線形補間
            var a = (timeTable[endIndex].ns - timeTable[startIndex].ns) / (timeTable[endIndex].size - timeTable[startIndex].size);
            var b = timeTable[startIndex].ns - a * timeTable[startIndex].size;

            var timeNanoSec = a * size + b;
            return new TimeSpan(timeNanoSec / 100);
        }
    }
}
