﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.FsFileCacheSimulator
{
    internal struct FileRegion : IEquatable<FileRegion>
    {
        public ulong Offset { get; }
        public ulong Size { get; }
        public FileRegion(ulong offset, ulong size)
        {
            Offset = offset;
            Size = size;
        }

        public static readonly FileRegion Zero = new FileRegion(0, 0);

        public ulong EndOffset => Offset + Size;

        public FileRegion ExpandAndAlign(ulong alignment)
        {
            var offset = BitUtility.AlignDown(Offset, alignment);
            var endOffset = BitUtility.AlignUp(EndOffset, alignment);
            return new FileRegion(offset, endOffset - offset);
        }

        public FileRegion ShrinkAndAlign(ulong alignment)
        {
            var offset = BitUtility.AlignUp(Offset, alignment);
            var endOffset = BitUtility.AlignDown(EndOffset, alignment);
            return new FileRegion(offset, endOffset - offset);
        }

        public static FileRegion GetInclusion(FileRegion region1, FileRegion region2)
        {
            var offset = Math.Min(region1.Offset, region2.Offset);
            var endOffset = Math.Max(region1.EndOffset, region2.EndOffset);
            return new FileRegion(offset, endOffset - offset);
        }

        public static FileRegion GetUnion(FileRegion region1, FileRegion region2)
        {
            if (!region1.Intersects(region2))
            {
                return Zero;
            }
            return GetInclusion(region1, region2);
        }

        public static FileRegion GetIntersection(FileRegion region1, FileRegion region2)
        {
            if (!region1.Intersects(region2))
            {
                return Zero;
            }
            var offset = Math.Max(region1.Offset, region2.Offset);
            var endOffset = Math.Min(region1.EndOffset, region2.EndOffset);
            return new FileRegion(offset, endOffset - offset);
        }

        public FileRegion GetEndRegionWithSizeLimit(ulong size)
        {
            if (size >= Size)
            {
                return this;
            }
            var offset = EndOffset - size;
            return new FileRegion(offset, size);
        }

        public bool Intersects(FileRegion other)
        {
            return Offset < other.EndOffset && other.Offset < EndOffset;
        }

        public bool Includes(FileRegion other)
        {
            return Offset <= other.Offset && other.EndOffset <= EndOffset;
        }

        public bool IncludesOffset(ulong offset)
        {
            return Offset <= offset && offset <= EndOffset;
        }

        public bool Equals(FileRegion other)
        {
            return Offset == other.Offset && Size == other.Size;
        }
        public override bool Equals(object obj)
        {
            if (obj is FileRegion)
            {
                return Equals((FileRegion)obj);
            }
            return false;
        }
        public override int GetHashCode()
        {
            return Offset.GetHashCode() ^ Size.GetHashCode();
        }
        public override string ToString()
        {
            return $"offset: {Offset}, size: {Size}";
        }
        public static bool operator ==(FileRegion lhs, FileRegion rhs)
        {
            return lhs.Equals(rhs);
        }
        public static bool operator !=(FileRegion lhs, FileRegion rhs)
        {
            return !lhs.Equals(rhs);
        }
    }
}
