﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.ToolFoundation.IO;

namespace Nintendo.FsFileCacheSimulator
{
    internal class CommandLineOption
    {
        [CommandLineValue(
            0,
            Description = "fs アクセスログ",
            IsRequired = true,
            ValueName = "path")]
        public string AccessLog { get; private set; }

        [CommandLineOption(
            "cache-size",
            Description = "キャッシュメモリのサイズ",
            DefaultValue = 128 * 1024 * 1024,
            ValueName = "bytes")]
        public ulong CacheSize { get; private set; }

        [CommandLineOption(
            "cache-size-mb",
            Description = "キャッシュメモリのサイズ (MB)",
            ValueName = "mbytes")]
        public ulong CacheSizeMB
        {
            get { return CacheSize >> 20; }
            private set { CacheSize = (value << 20); }
        }

        [CommandLineOption(
            "page-size",
            Description = "キャッシュメモリ内のページサイズ",
            DefaultValue = 16 * 1024,
            ValueName = "bytes")]
        public uint PageSize { get; private set; }

        [CommandLineOption(
            "page-size-kb",
            Description = "キャッシュメモリ内のページサイズ (KB)",
            ValueName = "kbytes")]
        public uint PageSizeKB
        {
            get { return PageSize >> 10; }
            private set { PageSize = (value << 10); }
        }

        [CommandLineOption(
            "readahead",
            Description = "一部のキャッシュアルゴリズムで先読みを行う",
            DefaultValue = true)]
        public bool EnableReadAhead { get; private set; }

        [CommandLineOption(
            "no-readahead",
            Description = "キャッシュアルゴリズムにおいて先読みを行わない")]
        public bool NoEnableReadAhead
        {
            get { return !EnableReadAhead; }
            private set { EnableReadAhead = !value; }
        }

        [CommandLineOption(
            "storage-type",
            Description = "シミュレーションに使用されるストレージの種類",
            DefaultValue = FileSystem.StorageType.Nand,
            ValueName = "enum")]
        public FileSystem.StorageType StorageType { get; private set; }

        [CommandLineOption(
            "nsp",
            Description = "fs アクセスログを生成したアプリケーションの nsp ファイル",
            ValueName = "path")]
        public string InputNspPath { get; private set; }

        [CommandLineOption(
            "original-nsp",
            Description = "fs アクセスログを生成したアプリケーションの v0 nsp ファイル",
            ValueName = "path")]
        public string OriginalNspPath { get; private set; }

        [CommandLineOption(
            "keyconfig",
            Description = "nsp ファイルのオープンに使用する keyconfig ファイル",
            ValueName = "path")]
        public string KeyConfigPath { get; private set; }

        [CommandLineOption(
            "split-result-per-s",
            Description = "シミュレーション結果を指定された秒ごとに分割して表示",
            ValueName = "seconds")]
        public ulong SplitResultPerSeconds
        {
            get { return SplitResultPerMilliseconds / 1000; }
            set { SplitResultPerMilliseconds = value * 1000; }
        }

        [CommandLineOption(
            "split-result-per-ms",
            Description = "シミュレーション結果を指定されたミリ秒ごとに分割して表示",
            ValueName = "milliseconds")]
        public ulong SplitResultPerMilliseconds { get; private set; }

        [CommandLineOption(
            "show-result-ipc-gt",
            Description = "指定された回数を超えて IPC を行った結果のみ表示",
            ValueName = "count",
            DefaultValue = -1)]
        public int ShowResultOfIpcGreaterThan { get; private set; }

        [CommandLineOption(
            "save-simulated-log",
            Description = "シミュレーションされたアクセスログを保存する",
            DefaultValue = false)]
        public bool SaveSimulatedLog { get; private set; }

        [CommandLineOption(
            "verbosity",
            Description = "ツールログの詳細度",
            DefaultValue = Log.Verbosity.Message,
            ValueName = "enum")]
        public Log.Verbosity Verbosity { get; private set; }

        public static CommandLineOption Parse(string[] args)
        {
            var parser = new CommandLineParser();

            CommandLineOption option;
            var continues = parser.ParseArgs(args, out option);

            if (continues)
            {
                return option;
            }
            else
            {
                return null;
            }
        }

        public static void CheckCommandLineOptionOrThrow(CommandLineOption option)
        {
            if (!File.Exists(option.AccessLog))
            {
                throw new ArgumentException($"アクセスログファイル '{option.AccessLog}' が存在しません");
            }
            if (option.CacheSize <= 0)
            {
                throw new ArgumentException("--cache-size には 0 よりも大きい値を指定してください");
            }
            if (option.PageSize <= 0)
            {
                throw new ArgumentException("--page-size には 0 よりも大きい値を指定してください");
            }
            if (option.CacheSize < option.PageSize)
            {
                throw new ArgumentException("--cache-size には --page-size よりも大きな値を指定してください");
            }
            if (BitUtility.ResetLeastSignificantBit1(option.PageSize) != 0)
            {
                throw new ArgumentException("--page-size には 2 のべき乗の値を指定してください");
            }
            if (!string.IsNullOrEmpty(option.InputNspPath) && !File.Exists(option.InputNspPath))
            {
                throw new ArgumentException($"nsp ファイル '{option.InputNspPath}' が存在しません");
            }
            if (!string.IsNullOrEmpty(option.OriginalNspPath) && !File.Exists(option.OriginalNspPath))
            {
                throw new ArgumentException($"nsp ファイル '{option.OriginalNspPath}' が存在しません");
            }
            if (!string.IsNullOrEmpty(option.KeyConfigPath) && !File.Exists(option.KeyConfigPath))
            {
                throw new ArgumentException($"keyconfig ファイル '{option.KeyConfigPath}' が存在しません");
            }
        }
    }
}
