<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Divide nsp.

    .DESCRIPTION
        Divide nsp into part files so that NX file system read them on FAT32 as a file.
#>

$scriptPath = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

$UNIT_SIZE = 4 * 1024 * 1024 * 1024 - 64 * 1024
$BUFFER_SIZE = 8 * 1024 * 1024
$BLOCK_SIZE = 32 * 1024

$global:progress = [long]0
$global:totalSize = [long]0

function Write-Progress
{
    $tmp = [System.String]::Format("{0, 3} %", [int] ([double] $global:progress / $global:totalSize * 100))
    [System.Console]::Write("{0}", $tmp)
    return 0
}

function Copy-Stream ([System.IO.Stream] $src, [System.IO.Stream] $dst, [long] $size, [Byte[]] $buffer)
{
    $restSize = $size
    while ($restSize -gt 0)
    {
        $copySize = [System.Math]::Min($restSize, $buffer.Length)
        $read = $src.Read($buffer, 0, $copySize)
        if ($read -ne $copySize)
        {
            return $read
        }
        $dst.Write($buffer, 0, $copySize)
        $restSize -= [long]$copySize
        $global:progress += [long]$copySize

        [System.Console]::SetCursorPosition([System.Console]::CursorLeft - 5, [System.Console]::CursorTop)
        Write-Progress | Out-Null
    }
    return $size
}

function Get-Index([long] $totalSize)
{
    $lastIndex = 0
    $restSize = $totalSize

    while ($restSize -gt 0)
    {
        $copySize = [System.Math]::Min($restSize, $UNIT_SIZE)
        $lastIndex += 1
        $restSize -= $copySize
    }

    return $lastIndex
}

foreach ($arg in $args)
{
    $fileInfo = New-Object System.IO.FileInfo($arg)
    if ($fileInfo.Length -lt $UNIT_SIZE)
    {
        [System.Console]::WriteLine("{0} is not need to be divided because its size is less than 0x{1:x} bytes.", $fileInfo.Name, $UNIT_SIZE)
        continue
    }
    $outputDir = $scriptDirectoryPath + "\" + $fileInfo.Name
    [System.Console]::WriteLine("source path: {0}, bytes: {1}", $outputDir, $fileInfo.Length)
    [System.IO.Directory]::CreateDirectory($outputDir) | Out-Null

    # アーカイブ属性を付与
    $dirInfo = New-Object System.IO.DirectoryInfo($outputDir)
    $dirInfo.Attributes = $dirInfo.Attributes -bor [System.IO.FileAttributes]::Archive

    $restSize = $fileInfo.Length
    $global:totalSize = $fileInfo.Length

    $index = 0
    $lastIndex = Get-Index $restSize

    $source = New-Object System.IO.FileStream($arg, [System.IO.FileMode]::Open, [System.IO.FileAccess]::Read, [System.IO.FileShare]::Read, $BLOCK_SIZE, [System.IO.FileOptions]::SequentialScan)
    if ($source -eq $null)
    {
        [System.Console]::WriteLine("Failed to open {0}.", $arg)
        exit 1
    }

    try
    {
        $buffer = New-Object byte[] $BUFFER_SIZE
        $copied = [long]0

        while ($restSize -gt 0)
        {
            $copySize = [System.Math]::Min($restSize, $UNIT_SIZE)
            $targetPath = [System.String]::Format("{0}\{1:d2}", $outputDir, $index)

            [System.Console]::Write("creating {0} ({1}/{2})... ", $targetPath, $index + 1, $lastIndex)
            Write-Progress | Out-Null

            $target = New-Object System.IO.FileStream($targetPath, [System.IO.FileMode]::Create, [System.IO.FileAccess]::Write, [System.IO.FileShare]::None, $BLOCK_SIZE, [System.IO.FileOptions]::SequentialScan)
            if ($target -eq $null)
            {
                [System.Console]::WriteLine("Failed to create {0}.", $targetPath)
                exit 1
            }
            try
            {
                $ret = Copy-Stream $source $target $copySize $buffer
                if ($ret -ne $copySize)
                {
                    [System.Console]::WriteLine("Failed to copy from {0} to {1}. {2}", $arg, $targetPath, $ret)
                    exit 1
                }
            }
            finally
            {
                $target.Dispose()
            }

            [System.Console]::SetCursorPosition([System.Console]::CursorLeft - 5, [System.Console]::CursorTop)
            [System.Console]::WriteLine("done.")

            $restSize -= $copySize
            $copied += [long]$copySize
            $index += 1
        }
    }
    finally
    {
        $source.Dispose()
    }

    [System.Console]::WriteLine("Success.")
}

exit 0