﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using DeviceTreeDataCollector;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakePartialDesc.Tests
{
    [TestClass()]
    public class MakePartialDescTests
    {
        private static string ListToString<T>(IEnumerable<T> list)
        {
            return "- " + string.Join("\n- ", list);
        }

        private class SanitizeMemoryMapsTestCase
        {
            public List<AddressRange> Input { get; set; }
            public List<AddressRange> ExpectedOutput { get; set; }

            public SanitizeMemoryMapsTestCase(List<AddressRange> input, List<AddressRange> expectedOutput)
            {
                Input = input;
                ExpectedOutput = expectedOutput;
            }

            public override string ToString()
            {
                return string.Format("Input: {0} / ExpectedOutput: {1}", ListToString(Input), ListToString(ExpectedOutput));
            }
        }

        [TestMethod()]
        public void SanitizeMemoryMapsTest()
        {
            var testCases = new List<SanitizeMemoryMapsTestCase>
            {
                new SanitizeMemoryMapsTestCase(
                    input: new List<AddressRange>
                    {
                        // 一致
                        new AddressRange(0x100000000, 0x20000000),
                        new AddressRange(0x100000000, 0x20000000),
                        new AddressRange(0x100000000, 0x20000000),

                        // 包含
                        new AddressRange(0x200000000, 0x20000000), // 頭が一致
                        new AddressRange(0x200000000, 0x40000000),
                        new AddressRange(0x200000000, 0x60000000),
                        new AddressRange(0x300000000, 0x40000000), // 末尾が一致
                        new AddressRange(0x320000000, 0x20000000),
                        new AddressRange(0x330000000, 0x10000000),
                        new AddressRange(0x400000000, 0x80000000), // 完全に包含
                        new AddressRange(0x420000000, 0x20000000),
                        new AddressRange(0x440000000, 0x10000000),
                        new AddressRange(0x440000000, 0x20000000),

                        // 合成可能
                        new AddressRange(0x500000000, 0x10000000), // 連続
                        new AddressRange(0x510000000, 0x20000000),
                        new AddressRange(0x530000000, 0x10000000),
                        new AddressRange(0x600000000, 0x20000000), // 部分重複
                        new AddressRange(0x610000000, 0x20000000),
                        new AddressRange(0x620000000, 0x20000000),
                    },
                    expectedOutput: new List<AddressRange>
                    {
                        new AddressRange(0x100000000, 0x20000000),
                        new AddressRange(0x200000000, 0x60000000),
                        new AddressRange(0x300000000, 0x40000000),
                        new AddressRange(0x400000000, 0x80000000),
                        new AddressRange(0x500000000, 0x40000000),
                        new AddressRange(0x600000000, 0x40000000),
                    }),
            };

            foreach (var testCase in testCases)
            {
                var output = MakePartialDesc.SanitizeMemoryMaps(testCase.Input);
                Assert.IsTrue(testCase.ExpectedOutput.SequenceEqual(output),
                    "Expected:\n" + ListToString(testCase.ExpectedOutput) + "\n !=\nActual:\n" + ListToString(output));

                var outputFromReversed = MakePartialDesc.SanitizeMemoryMaps(testCase.Input.Reverse<AddressRange>());
                Assert.IsTrue(testCase.ExpectedOutput.SequenceEqual(outputFromReversed),
                    "Expected:\n" + ListToString(testCase.ExpectedOutput) + "\n !=\nActual:\n" + ListToString(outputFromReversed));

                foreach (int loop in Enumerable.Range(1, 10))
                {
                    var shuffledInput = testCase.Input.OrderBy(i => Guid.NewGuid()).ToList();
                    Console.WriteLine("shuffledInput:\n{0}", ListToString(shuffledInput));
                    var outputFromShuffled = MakePartialDesc.SanitizeMemoryMaps(shuffledInput);
                    Assert.IsTrue(testCase.ExpectedOutput.SequenceEqual(outputFromShuffled),
                        "Expected:\n" + ListToString(testCase.ExpectedOutput) + "\n !=\nActual:\n" + ListToString(outputFromShuffled));
                }
            }
        }
    }
}
