﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Devkit setup support script.

    .DESCRIPTION
        This script supports setup of Devkits.
        This scirpt uses offical setup tools, so more safety.

    .PARAMETER TargetsSdev
        Specifies SDEV's IP address or serial number.
        By using comma, multi SDEV can be specified.

    .PARAMETER TargetsEdev
        Specifies EDEV's serial number.
        By using comma, multi EDEV can be specified.

    .PARAMETER ConnectSdevAll
        All connected SDEV are setup.

    .PARAMETER ConnectEdevAll
        All connected EDEV are setup.

    .PARAMETER SdevTryConcurrent
        Try to setup SDEV concurrently.
        (This is trial function. This helps rapid setup, but sometimes becomes fail.)

    .EXAMPLE
        Setup specified SDEV by serial number.

        > DevKitSetup.ps1 -TargetsSdev XAW02121000001,XAW02121000002

    .EXAMPLE
        Setup specified SDEV by IP.

        > DevKitSetup.ps1 -TargetsSdev 192.168.10.10,192.168.10.11

    .EXAMPLE
        Setup specified SDEV by MAC address.

        > DevKitSetup.ps1 -TargetsSdev 7C:BB:8A:29:F5:0F

    .EXAMPLE
        Setup specified EDEV by serial number.

        > DevKitSetup.ps1 -TargetsEdev XAL03100000001,XAL03100000002

    .EXAMPLE
        Setup both SDEV and EDEV.

        > DevKitSetup.ps1 -TargetsSdev XAW02121000001 -TargetsEdev XAL03100000001

    .EXAMPLE
        Setup all connected SDEV to PC.

        > DevKitSetup.ps1 -ConnectSdevAll

    .EXAMPLE
        Setup all connected EDEV to PC.

        > DevKitSetup.ps1 -ConnectEdevAll

#>

param(
    [string[]] $TargetsSdev = @(),
    [string[]] $TargetsEdev = @(),
    [switch] $ConnectSdevAll,
    [switch] $ConnectEdevAll,
    [switch] $SdevTryConcurrent
)

$SetupSdevArray = @()
$SetupEdevArray = @()

$InitializeSdevPath = ""
$InitializeEdevPath = ""
$InitializeSdevRelPath = "Tools\CommandLineTools\InitializeSdev.exe"
$InitializeEdevRelPath = "Tools\CommandLineTools\InitializeEdev.exe"

function SetSdkRoot()
{
    $scriptDirectoryPath = Split-Path $script:MyInvocation.MyCommand.Path -Parent
    $rootDirectory = Split-Path -Parent $scriptDirectoryPath | Split-Path -Parent
    if (Join-Path $rootDirectory NintendoSdkRootMark | Test-Path)
    {
        Set-Item env:NINTENDO_SDK_ROOT -value $rootDirectory
    } else {
        $rootDirectory = Split-Path -Parent $rootDirectory | Split-Path -Parent | Split-Path -Parent
        if (Join-Path $rootDirectory NintendoSdkRootMark | Test-Path)
        {
            Set-Item env:NINTENDO_SDK_ROOT -value $rootDirectory
        }
    }
    $script:InitializeSdevPath = Join-Path $env:NINTENDO_SDK_ROOT $InitializeSdevRelPath
    $script:InitializeEdevPath = Join-Path $env:NINTENDO_SDK_ROOT $InitializeEdevRelPath
}

function SpecifyDevKitNamesFromArguments()
{
    if( $SdevTryConcurrent )
    {
        Write-Host "[WARN] -SdevTryConcurrent is trial option. This option sometimes occurs failure."
    }

    if( $TargetsSdev -ne @() -or $TargetsEdev -ne @() )
    {
        $script:SetupSdevArray += $TargetsSdev
        $script:SetupEdevArray += $TargetsEdev
        Return
    }
    if( $ConnectSdevAll )
    {
        $commandLog = & $InitializeSdevPath "--list-available-targets" | Out-String -Stream
        $script:SetupSdevArray += $commandLog
    }
    if( $ConnectEdevAll )
    {
        $commandLog = & $InitializeEdevPath "--list-available-targets" | Out-String -Stream
        $script:SetupEdevArray += $commandLog
    }
}

function InitializeDevKits()
{
    if( -Not($SdevTryConcurrent) )
    {
        foreach($target in $script:SetupSdevArray){
            & $InitializeSdevPath "--target" "$target" "--verbose"
            if( $LASTEXITCODE -ne 0 )
            {
                Write-Error "InitializeSdev $commandArgument Failed."
            }
            else
            {
                Write-Host "InitializeSdev $commandArgument succeeded."
            }
        }
    }
    else
    {
        $procArray = @()
        foreach ($target in $script:SetupSdevArray){
            $commandArgument = " --target {0} {1} --verbose" -f $target, "--keep-targetmanager-alive"
            $procArray += Start-Process -FilePath $InitializeSdevPath -ArgumentList $commandArgument -PassThru -WindowStyle Minimized
        }
        foreach ($proc in $procArray)
        {
            if(!$proc.HasExited)
            {
                Wait-Process -InputObject $proc -Timeout 600
            }
        }
        foreach ($proc in $procArray)
        {
            if($proc.ExitCode -ne 0)
            {
                Write-Error "InitializeSdev Failed."
            }
            else
            {
                Write-Host "InitializeSdev succeeded."
            }
        }
    }

    foreach($target in $script:SetupEdevArray){
        $commandArgument = " --target {0} {1} --verbose" -f $target, ""
        & $InitializeEdevPath "--target" "$target" "--verbose"
        if( $LASTEXITCODE -ne 0 )
        {
            Write-Error "InitializeEdev $commandArgument Failed."
        }
        else
        {
            Write-Host "InitializeEdev $commandArgument succeeded."
        }
    }
}

#######################################################################
#
# Main Start
#
#######################################################################

# Setting NINTENDO_SDK_ROOT environment variable.
SetSdkRoot

# Specifiy devkits to setup from arguments to variable $SetupSdevArray and $SetupEdevArray.
SpecifyDevKitNamesFromArguments

# Initialize devkits of $SetupSdevArray, $SetupEdevArray.
InitializeDevKits

# Log about finishing this script.
Write-Host $MyInvocation.MyCommand.Name "FINISHED."
exit 0

