﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.Serialization.Json;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace TeamCityLogParser
{
    public class LogParser
    {
        public static string ParseToJsonText(string filepath)
        {
            return ParseToJsonTextFromText(File.ReadAllText(filepath));
        }

        public static string ParseToJsonTextFromText(string text)
        {
            var date = ParseDate(text);
            var revision = ParseRevision(text);
            var branch = ParseBranch(text);

            var dictionary = new Dictionary<string, object>()
            {
                { "date", date.ToString() },
                { "revision", revision },
                { "branch", branch },
            };

            return ConvertToJsonText(dictionary);
        }

        /// <summary>
        /// 対象の型を Json のテキスト情報に変換します。
        /// </summary>
        /// <param name="data"></param>
        /// <returns></returns>
        public static string ConvertToJsonText(Dictionary<string, object> data)
        {
            var settings = new DataContractJsonSerializerSettings();
            settings.UseSimpleDictionaryFormat = true;

            var serializer = new DataContractJsonSerializer(typeof(Dictionary<string, object>), settings);
            var stream = new MemoryStream();
            serializer.WriteObject(stream, data);

            return Encoding.UTF8.GetString(stream.ToArray());
        }

        /// <summary>
        /// "Started 'Wed Apr 09 17:15:44 JST 2014" 形式の日付情報をパースします。
        /// </summary>
        /// <param name="text"></param>
        /// <returns></returns>
        public static DateTime ParseDate(string text)
        {
            var match = Regex.Match(text, @"Started '(.*)' on '.*' by '.*'");

            if (!match.Success)
            {
                throw new ArgumentException("日付を取得できません。");
            }

            var date = match.Groups[1].Value.Replace("JST", "+09:00");

            return DateTime.ParseExact(
                date,
                "ddd MMM dd HH:mm:ss zzz yyyy",
                System.Globalization.DateTimeFormatInfo.InvariantInfo,
                System.Globalization.DateTimeStyles.None);
        }

        /// <summary>
        /// "[17:15:43]:     [Available server properties] system.build.vcs.number.1 = df32428f53aad9a17f3797a637da04b92147afcd" 形式の
        /// リビジョン情報をパースします。
        /// </summary>
        /// <param name="text"></param>
        /// <returns></returns>
        public static string ParseRevision(string text)
        {
            var match = Regex.Match(text, @"\[.*\]:\s+\[Available server properties\] system\.build\.vcs\.number\.1 = ([0-9a-f]+)");

            if (!match.Success)
            {
                throw new ArgumentException("リビジョンを取得出来ません。");
            }

            return match.Groups[1].Value;
        }

        /// <summary>
        /// "[09:16:01]:     [Available server properties] teamcity.build.vcs.branch.SigloSdk_SigloSdk = refs/heads/feature/SIGLO-1100_modifying_nw4f_sln" 形式の
        /// ブランチ情報をパースします。
        /// </summary>
        /// <param name="text"></param>
        /// <returns></returns>
        public static string ParseBranch(string text)
        {
            var match = Regex.Match(text, @"\[.*\]:\s+\[Available server properties\] teamcity\.build\.vcs\.branch\.SigloSdk_SigloSdk\s+=\s+([^\s]+)");

            if (!match.Success)
            {
                throw new ArgumentException("リビジョンを取得出来ません。");
            }

            return match.Groups[1].Value;
        }
    }
}
