﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.Serialization;
using ContentsUploader.Assistants;

namespace ContentsUploader.Models
{
    using static Constants;

    public static partial class Pms
    {
        public interface ILanguageItem<T>
        {
            string lang { get; set; }
            string value { get; set; }
        }

        // 不足している言語設定を作成
        private static List<T> BuildLanguageItems<T>(List<T> items, Languages languages) where T : ILanguageItem<T>, new()
        {
            // 設定済み言語の参照マップを作成
            var map = new Dictionary<string, string>();
            if (items != null)
            {
                foreach (var item in items)
                {
                    map.Add(item.lang, item.value);
                    languages |= ToolUtility.ToEnumValue<Languages>(item.lang, LanguagesNone);
                }
            }

            // 設定済みと指定された言語の両方を含むリストを生成
            var results = new List<T>();
            foreach (var language in ToolUtility.ToEnumNames(languages))
            {
                var item = new T();
                item.lang = language;
                if (map.ContainsKey(language))
                {
                    // 設定済みなら値を引き継ぐ
                    item.value = map[language];
                    map.Remove(language);
                }
                else
                {
                    // 新規なら空の値を設定
                    item.value = "";
                }
                results.Add(item);
            }

            // 下記で生成されるのは、enum Languages に定義されていない言語
            foreach (var language in map.Keys)
            {
                Log.WriteLine($"Warning: Unknown language code \"{language}\".");
                var item = new T();
                item.lang = language;
                item.value = map[language];
                results.Add(item);
            }
            return results;
        }

        // 指定方法で言語設定を変更
        private static void UpdateLanguageItems<T>(List<T> items, Languages languages, Func<string, string> assign, bool isValueChanged, bool isLaguageChanged) where T : ILanguageItem<T>
        {
            var target = ToolUtility.ToEnumNames(languages);
            if (isValueChanged || isLaguageChanged)
            {
                // 処理定義（指定により異なる）
                Action<T, string> update;
                if (isValueChanged && isLaguageChanged)
                {
                    update = (item, value) =>
                    {
                        // 両方の変更指示なら、対象言語の値のみ置き換えて、対象外の値はクリア
                        item.value = target.Contains(item.lang) ? value : "";
                    };
                }
                else if (isValueChanged)
                {
                    update = (item, value) =>
                    {
                        // 値のみの変更指示なら、リストにある全言語の値を置き換える
                        item.value = value;
                    };
                }
                else if (isLaguageChanged)
                {
                    update = (item, value) =>
                    {
                        // 言語のみの変更指示なら
                        if (target.Contains(item.lang))
                        {
                            // 対象言語かつ値が空なら置き換えて、空以外なら変更しない
                            if (string.IsNullOrEmpty(item.value))
                            {
                                item.value = value;
                            }
                        }
                        else
                        {
                            // 対象言語でなければ値をクリア
                            item.value = "";
                        }
                    };
                }
                else
                {
                    // 変更しない
                    update = (formaName, value) => { };
                }

                // 更新処理
                foreach (var item in items)
                {
                    update(item, assign(item.lang));
                }
            }
        }

        // タイプ・ステータス取得のみで利用
        [DataContract]
        public class ConfirmInfo
        {
            [DataMember(Name = "type")]
            public string type { get; set; }

            [DataMember(Name = "status")]
            public string status { get; set; }
        }

        // ステータス取得のみで利用
        [DataContract]
        public class StatusInfo
        {
            [DataMember(Name = "status")]
            public string status { get; set; }
        }

        // 更新時間取得のみで利用
        [DataContract]
        public class LastUpdateDatetimeInfo
        {
            [DataMember(Name = "last_update_datetime")]
            public long lastUpdateDatetime { get; set; }
        }

        // NsUid 取得のみで利用
        [DataContract]
        public class NsUidInfo
        {
            [DataMember(Name = "id")]
            public long id { get; set; }

            [DataMember(Name = "last_update_datetime")]
            public long lastUpdateDatetime { get; set; }
        }
    }
}
