﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using ContentsUploader.Assistants;

namespace ContentsUploader
{
    using static Constants;
    using static Models.Star;
    using static WebAccessor.Method;

    // ログ出力
    public static class Log
    {
        public static string Prefix { get;  set; } = "[ContentsUploader]";
        public static bool WithTrace { get;  set; } = false;

        public static void WriteLine(string message)
        {
            var prefix = string.IsNullOrEmpty(Prefix) ? "" : Prefix + " ";
            Console.WriteLine(prefix + message);
            if (WithTrace)
            {
                System.Diagnostics.Trace.WriteLine(prefix + message);
            }
        }

        public static void WriteLine(string format, params object[] args)
        {
            var prefix = string.IsNullOrEmpty(Prefix) ? "" : Prefix + " ";
            var message = string.Format(prefix + format, args);
            Console.WriteLine(message);
            if (WithTrace)
            {
                System.Diagnostics.Trace.WriteLine(message);
            }
        }

        public static void WriteLineAsIs(string message)
        {
            Console.WriteLine(message);
            if (WithTrace)
            {
                System.Diagnostics.Trace.WriteLine(message);
            }
        }

        public static void WriteLineAsIs(string format, params object[] args)
        {
            var message = string.Format(format, args);
            Console.WriteLine(message);
            if (WithTrace)
            {
                System.Diagnostics.Trace.WriteLine(message);
            }
        }
    }

    // ID（ApplicationId、ContentMetaId、TitleId など）
    public sealed class Id64 : IComparable, IComparable<Id64>, IEquatable<Id64>
    {
        private const ulong InvalidValue = 0;
        public static readonly Id64 Invalid = new Id64(InvalidValue);

        private const ulong QuestMenuProgramValue = 0x0100069000078000;
        public static readonly Id64 QuestMenuProgram = new Id64(QuestMenuProgramValue);

        public static readonly Id64[] Unregistrable = {
                Invalid,
            };

        private const ulong AocIdOffset = 0x01000;
        private const ulong AocCountMax = 0x00FFF;

        public static int ToAocIndex(Id64 applicationId, Id64 contentMetaId)
        {
            int index = 0;
            ulong min = applicationId.Value + AocIdOffset;
            ulong max = min + AocCountMax;
            ulong val = contentMetaId.Value;
            if (val >= min && val <= max)
            {
                index = (int)(val - min);
            }
            return index;
        }

        public ulong Value { get; }
        public string FormalStyle { get; }
        public string PrefixedStyle { get; }
        public bool IsValid { get; }

        public Id64(ulong value)
        {
            Value = value;

            FormalStyle = Value.ToString("x16");
            PrefixedStyle = $"0x{FormalStyle}";
            IsValid = InvalidValue != Value;
        }

        public Id64(string value, int radix = 16)
            : this(System.Convert.ToUInt64(value, radix))
        {
        }

        public override string ToString()
        {
            return FormalStyle;
        }

        public override int GetHashCode()
        {
            return Value.GetHashCode();
        }

        public int CompareTo(object obj)
        {
            return CompareTo(obj as Id64);
        }

        public int CompareTo(Id64 other)
        {
            if (object.ReferenceEquals(other, null))
            {
                return 1;
            }
            return Value.CompareTo(other.Value);
        }

        public override bool Equals(object obj)
        {
            return Equals(obj as Id64);
        }

        public bool Equals(Id64 other)
        {
            if (object.ReferenceEquals(other, null))
            {
                return false;
            }
            if (object.ReferenceEquals(other, this))
            {
                return true;
            }
            return Value.Equals(other.Value);
        }

        public static bool operator ==(Id64 lhs, Id64 rhs)
        {
            if (object.ReferenceEquals(lhs, null))
            {
                return object.ReferenceEquals(rhs, null);
            }
            return lhs.Equals(rhs);
        }

        public static bool operator !=(Id64 lhs, Id64 rhs)
        {
            return !(lhs == rhs);
        }
    }

    // サービス環境
    public class ServiceEnvironment : IEquatable<ServiceEnvironment>
    {
        private static readonly ServiceEnvironment Invalid = new ServiceEnvironment("", "");
        public static readonly ServiceEnvironment Dev1 = new ServiceEnvironment("dev1", "jd1");
        public static readonly ServiceEnvironment Dev6 = new ServiceEnvironment("dev6", "td1");

        public string Name { get; }     // service name. e.g. dev6, dev1
        public string Alias { get; }    // discovery alias name. e.g. td1, jd1
        public bool IsValid { get; }

        private ServiceEnvironment(string name, string alias)
        {
            Name = name.ToLower();
            Alias = alias.ToLower();

            IsValid = !string.IsNullOrEmpty(Name);
        }

        public override string ToString()
        {
            return Name;
        }

        public override int GetHashCode()
        {
            return Name.GetHashCode();
        }

        public override bool Equals(object obj)
        {
            return Equals(obj as ServiceEnvironment);
        }

        public bool Equals(ServiceEnvironment other)
        {
            if (object.ReferenceEquals(other, null))
            {
                return false;
            }
            if (object.ReferenceEquals(other, this))
            {
                return true;
            }
            return Name.Equals(other.Name) && Alias.Equals(other.Alias);
        }

        public static bool operator ==(ServiceEnvironment lhs, ServiceEnvironment rhs)
        {
            if (object.ReferenceEquals(lhs, null))
            {
                return object.ReferenceEquals(rhs, null);
            }
            return lhs.Equals(rhs);
        }

        public static bool operator !=(ServiceEnvironment lhs, ServiceEnvironment rhs)
        {
            return !(lhs == rhs);
        }

        public static ServiceEnvironment ConvertFromAlias(string alias)
        {
            foreach (var env in new ServiceEnvironment[] { Dev1, Dev6 })
            {
                if (string.Compare(env.Alias, alias, true) == 0)
                {
                    return env;
                }
            }
            return Invalid;
        }
    }

    // サービス API
    public class ServiceApi
    {
        // D4C Server
        protected string Osiris { get; }
        protected string Star { get; }
        protected string Superfly { get; }

        // eShop Server
        protected string Ashigaru { get; }
        protected string Pms { get; }
        protected string Smt { get; }

        // D4C API
        public string OsirisV1 { get; }
        public string StarV1 { get; }
        public string SuperflyV1 { get; }

        // eShop API
        public string ShogunV1 { get; }
        public string PmsRestV2 { get; }
        public string SmtRs { get; }

        public ServiceApi(ServiceEnvironment environment)
        {
            // サーバー名定義
            Osiris = $"https://osiris.ws.{environment.Name}.d4c.nintendo.net";
            Star = $"https://star.debug.{environment.Name}.d4c.nintendo.net";
            Superfly = $"https://superfly-b-d.ws.{environment.Name}.d4c.nintendo.net";

            Ashigaru = $"https://ashigaru-frontend.hac.{environment.Alias}.eshop.nintendo.net";
            Pms = $"https://pms.wc.{environment.Name}.eshop.nintendo.net";
            Smt = $"https://smt.wc.{environment.Name}.eshop.nintendo.net";

            // 共通部分定義（終端には / を含めない）
            OsirisV1 = $"{Osiris}/v1";
            StarV1 = $"{Star}/v1";
            SuperflyV1 = $"{Superfly}/v1";

            ShogunV1 = $"{Ashigaru}/shogun/v1";
            PmsRestV2 = $"{Pms}/pms/rest/v2";
            SmtRs = $"{Smt}/smt/rs";
        }
    }

    // トークン情報
    public class Token
    {
        public static readonly Token Invalid = new Token();

        public string Path { get; }
        public TokenInfo Contents { get; }
        public bool IsValid { get; }

        private Token()
        {
            Path = "";
            Contents = new TokenInfo();
            IsValid = false;
        }

        public Token(string path)
        {
            var contents = System.IO.File.ReadAllText(path);

            Path = path;
            Contents = ToolUtility.Deserialize<TokenInfo>(contents);
            IsValid = !string.IsNullOrEmpty(Contents.accessToken);
        }

        public override string ToString()
        {
            return ToolUtility.Serialize<TokenInfo>(Contents);
        }
    }

    // 名前付け情報
    public class Naming
    {
        public static readonly Naming Invalid = new Naming("", "", "", "");

        public string InitialCode { get; }
        public string InitialPrefix { get; }
        public string InitialSuffix { get; }
        public string FormalName { get; }

        public bool IsValid { get; }
        public bool HasFormalName { get; }

        public static Naming CreateTitle(string initialCode, string formalName)
        {
            return new Naming(initialCode, "HAC_", "", formalName);
        }

        public static Naming CreateDemo(string formalName)
        {
            return new Naming("PSG01", "HACP", " %application_id%", formalName);
        }

        public Naming CreateDefault()
        {
            return new Naming(InitialCode, InitialPrefix, InitialSuffix, "");
        }

        private Naming(string initialCode, string initialPrefix, string initialSuffix, string formalName)
        {
            InitialCode = initialCode;
            InitialPrefix = initialPrefix;
            InitialSuffix = initialSuffix;
            FormalName = formalName;

            IsValid = ToolUtility.ValidateInitialCode(InitialCode);
            HasFormalName = !string.IsNullOrEmpty(FormalName);
        }

        public string ToFormalName(Id64 applicationId, string language, int index = 0)
        {
            // 正式名称がなければ、イニシャルコードを用いる
            var name = HasFormalName ? FormalName : $"{InitialPrefix}{InitialCode}{InitialSuffix}";
            name = name.Replace("%initial_code%", InitialCode);
            name = name.Replace("%application_id%", applicationId.FormalStyle);
            name = name.Replace("%language%", language);
            name = name.Replace("%language_name%", ToLanguageName(language));
            name = name.Replace("%index%", index.ToString());
            return name;
        }
    }

    // ContentsUploader 設定
    public class Setting : System.IDisposable
    {
        public static Setting Current { get; private set; }

        public static bool CreateCurrent(string alias, string porxy, bool verbose)
        {
            Current = new Setting(ServiceEnvironment.ConvertFromAlias(alias), porxy, verbose);
            if (!Current.Environment.IsValid)
            {
                Log.WriteLine($"Error: Please input valid value by --environment.");
                return false;
            }
            return true;
        }

        public DateTime StartDateTime { get; }
        public ServiceEnvironment Environment { get; }
        public string Proxy { get; }
        public bool IsVerbose { get; }

        public ServiceApi Api { get; }
        public Token Token { get; private set; }
        public Naming Naming { get; private set; }
        public Languages Languages { get; private set; }
        public bool HasLanguages { get { return Languages != LanguagesNone; } }
        public List<string> TargetCountries { get; private set; }
        public bool HasTargetCountries { get; private set; }
        public CommandUtility.TemporaryFileHolder FileHolder { get; private set; }
        private bool IsDisposed { get; set; }

        private Setting(ServiceEnvironment environment, string proxy, bool verbose)
        {
            StartDateTime = DateTime.Now;
            Environment = environment;
            Proxy = string.IsNullOrEmpty(proxy) ? "" : proxy;
            IsVerbose = verbose;

            Api = new ServiceApi(environment);
            Token = Token.Invalid;
            Naming = Naming.Invalid;
            Languages = LanguagesNone;
            TargetCountries = new List<string>();
            HasTargetCountries = false;
            FileHolder = new CommandUtility.TemporaryFileHolder("ContentsUploader");
            IsDisposed = false;
        }

        ~Setting()
        {
            Dispose(false);
        }

        public void Dispose()
        {
            Dispose(true);
        }

        private void Dispose(bool disposing)
        {
            if (!IsDisposed)
            {
                if (disposing)
                {
                    // managed object の解放
                }

                // unmanaged object の解放
                if (FileHolder != null)
                {
                    // TemporaryFileHolder は、自身のデストラクタで Dispose しないので
                    // ここで一時ファイルの破棄処理を行っておく
                    FileHolder.Dispose();
                    FileHolder = null;
                }
                IsDisposed = true;
            }
        }

        public bool PublishToken(string token, string user, string password)
        {
            var contents = string.Empty;
            if (!PublishTokenByOption(out contents, token, user, password))
            {
                return false;
            }

            try
            {
                var dir = FileHolder.CreateTemporaryDirectory("token");
                var path = System.IO.Path.Combine(dir.FullName, "oauth_token.json");
                System.IO.File.WriteAllText(path, contents);
                Token = new Token(path);
            }
            catch (System.Exception e)
            {
                Log.WriteLine($"Exception: {e.Message}");
                return false;
            }
            return true;
        }

        private bool PublishTokenByOption(out string contents, string token, string user, string password)
        {
            contents = string.Empty;
            if (!string.IsNullOrEmpty(token))
            {
                // token オプションによる指定
                var path = string.Empty;
                if (!ToolUtility.ConvertToAbsoluteFilePath(out path, token))
                {
                    Log.WriteLine($"Error: File not found \"{path}\".");
                    return false;
                }
                contents = System.IO.File.ReadAllText(path);
            }
            else if (!string.IsNullOrEmpty(user) && !string.IsNullOrEmpty(password))
            {
                // user password オプションによる指定
                var web = new WebAccessor(Proxy, false);
                var uri = new Uri($"{Api.StarV1}/ndid/users/@me/token");
                var account = new Account(user, password);
                if (!web.BasicAuthentication(out contents, uri, account))
                {
                    Log.WriteLine($"Error: Basic authentication failed.");
                    return false;
                }
            }
            else
            {
                Log.WriteLine($"Error: Please input (--token) or (--user, --password).");
                return false;
            }
            return true;
        }

        public bool PublishNewPassword()
        {
            if (!Token.IsValid)
            {
                Log.WriteLine($"Error: Your token file doesn't have access_token value.");
                return false;
            }

            var token = Token.Contents.accessToken;
            var web = new WebAccessor(Proxy, true);
            var uri = new Uri($"{Api.StarV1}/ndid/users?access_token={token}");
            return web.HttpRequest(POST, uri);
        }

        public bool SetupNaming(Naming naming)
        {
            if (!naming.IsValid)
            {
                Log.WriteLine($"Error: Invalid initial code. Please input 5 alphanumeric characters.");
                return false;
            }
            Naming = naming;
            return true;
        }

        public bool SetupDemoLanguages(string options)
        {
            return SetupLanguages(options, DemoLanguagesRequired, DemoLanguagesDefault, DemoLanguagesAll);
        }

        public bool SetupTitleLanguages(string options)
        {
            return SetupLanguages(options, TitleLanguagesRequired, TitleLanguagesDefault, TitleLanguagesAll);
        }

        private bool SetupLanguages(string options, Languages required, Languages defaults, Languages all)
        {
            var languages = LanguagesNone;

            // オプションを解析
            var delimiter = new char[] { ',' };
            foreach (var option in options.Split(delimiter, StringSplitOptions.RemoveEmptyEntries))
            {
                var value = LanguagesNone;
                if (string.Compare(option, "required", true) == 0)
                {
                    value = required;
                }
                else if (string.Compare(option, "default", true) == 0)
                {
                    value = defaults;
                }
                else if (string.Compare(option, "all", true) == 0)
                {
                    value = all;
                }
                else
                {
                    value = ToolUtility.ToEnumValue<Languages>(option, LanguagesNone);
                }

                if (value == LanguagesNone)
                {
                    Log.WriteLine($"Error: Invalid language. \"{option}\" is not unsupported.");
                    return false;
                }
                else
                {
                    languages |= value;
                }
            }

            // 何らかの言語が指定されていれば必須言語を追加
            if (languages != LanguagesNone)
            {
                languages |= required;
            }
            Languages = languages;
            return true;
        }

        public bool SetupTitleCountries(string options)
        {
            return SetupTargetCountries(options, TitleCountries);
        }

        private bool SetupTargetCountries(string options, List<string> defaults)
        {
            // オプションを解析
            var specifed = false;
            var preset = false;
            var delimiter = new char[] { ',' };
            var countries = new Dictionary<string, Country>();
            foreach (var option in options.Split(delimiter, StringSplitOptions.RemoveEmptyEntries))
            {
                if (string.Compare(option, "default", true) == 0)
                {
                    preset = true;
                }
                else if (Countries.ContainsKey(option))
                {
                    var country = Countries[option];
                    countries[country.Name] = country;
                    specifed = true;
                }
                else
                {
                    Log.WriteLine($"Error: Invalid country. \"{option}\" is not unsupported.");
                    return false;
                }
            }
            if (preset || countries.Count == 0)
            {
                foreach (var code in defaults)
                {
                    var country = Countries[code];
                    countries[country.Name] = country;
                }
            }

            // 登録する際は国名順なので、名前でソートしてコードを追加
            var codes = new List<string>();
            var names = new List<string>(countries.Keys);
            names.Sort();
            foreach (var name in names)
            {
                codes.Add(countries[name].Code);
            }
            TargetCountries = codes;
            HasTargetCountries = specifed;
            return true;
        }
    }
}
