﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using Nintendo.Foundation.IO;
using ContentsUploader.Assistants;

namespace ContentsUploader.Commands
{
    public class RegisterVersionCommand : CommandBase
    {
        //! Nsp 入力
        [CommandLineOption('s', "input", DefaultValue = "", Description = "Nsp of register-version setting")]
        public string NspPathNewOption { get; set; }

        //! Nsp 入力 (旧オプション)
        [CommandLineOption("nsp", DefaultValue = "", IsHidden = true, Description = "Nsp of register-version setting")]
        public string NspPathOldOption { get; set; }

        //! 4 変数入力 ->
        [CommandLineOption("application-id", DefaultValue = "", Description = "Application ID of register-version setting")]
        public override string ApplicationIdOption { get; set; }

        [CommandLineOption("content-meta-id", DefaultValue = "", Description = "Content Meta ID of register-version setting")]
        public override string ContentMetaIdOption { get; set; }

        [CommandLineOption("type", DefaultValue = "", Description = "Type of register-version setting")]
        public string ContentMetaTypeOption { get; set; }

        [CommandLineOption("content-meta-version", DefaultValue = "", Description = "Delivery Version of register-version setting")]
        public string ContentMetaVersionOption { get; set; }
        //! <-

        public override string NspPathOption
        {
            get { return !string.IsNullOrEmpty(NspPathNewOption.Trim()) ? NspPathNewOption.Trim() : NspPathOldOption.Trim(); }
            set { NspPathNewOption = value; }
        }

        //! 通知バージョン
        [CommandLineOption("content-meta-notify-version", DefaultValue = "", Description = "Notify Version of register-version setting")]
        public string NotifyVersionOption { get; set; }

        // ApplicationId, ContentMetaId 指定の有無
        protected bool HasContentMetaSet()
        {
            return !string.IsNullOrEmpty(ApplicationIdOption) && !string.IsNullOrEmpty(ContentMetaIdOption);
        }

        public override void Run()
        {
            Run("Register Version");
        }

        protected override bool RunByFile(NspAccessor nsp)
        {
            var result = true;
            foreach (var content in nsp.NspInnerContent)
            {
                var deliveryVersion = string.IsNullOrEmpty(ContentMetaVersionOption) ? content.Version : int.Parse(ContentMetaVersionOption);
                var notifyVersion = string.IsNullOrEmpty(NotifyVersionOption) ? content.Version : int.Parse(NotifyVersionOption);
                result &= RegisterVersion(nsp.ApplicationId, content.ContentMetaId, content.Type, deliveryVersion, notifyVersion);
            }
            return result;
        }

        protected override bool RunByContentMetaId()
        {
            return RegisterVersion();
        }

        protected override bool RunByApplicationId()
        {
            return RegisterVersion();
        }

        private bool RegisterVersion()
        {
            var applicationId = new Id64(ApplicationIdOption);
            var contentMetaId = new Id64(ContentMetaIdOption);
            var contentMetaType = ToolUtility.ToEnumValue<ContentMetaType>(ContentMetaTypeOption);
            var deliveryVersion = int.Parse(ContentMetaVersionOption);
            var notifyVersion = string.IsNullOrEmpty(NotifyVersionOption) ? deliveryVersion : int.Parse(NotifyVersionOption);
            return RegisterVersion(applicationId, contentMetaId, contentMetaType, deliveryVersion, notifyVersion);
        }

        private bool RegisterVersion(Id64 applicationId, Id64 contentMetaId, ContentMetaType contentMetaType, int deliveryVersion, int notifyVersion)
        {
            Log.WriteLine($"Register Version: {applicationId}, {contentMetaId}, {contentMetaType}, {deliveryVersion}, {notifyVersion}");

            var d4c = new D4cHelper(Setting.Current);
            return d4c.RegisterVersionByContentMeta(applicationId, contentMetaId, contentMetaType, deliveryVersion, notifyVersion);
        }
    }
}
