﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using Nintendo.Foundation.IO;
using ContentsUploader.Assistants;

namespace ContentsUploader.Commands
{
    using static Constants;
    using static Models.Smt;

    public abstract class PricingCommand : CommandBase
    {
        [CommandLineOption("initial-code", DefaultValue = "", Description = "initial code", IsRequired = true)]
        public string InitialCodeOption { get; set; }

        [CommandLineOption("name", DefaultValue = "", Description = "item name")]
        public string FormalNameOption { get; set; }

        [CommandLineOption("language", DefaultValue = "", Description = "item language")]
        public string LanguageOption { get; set; }

        [CommandLineOption("target-country", DefaultValue = "", Description = "item country")]
        public string TargetCountryOption { get; set; }

        [CommandLineOption("price-control", DefaultValue = "", Description = "Price control option")]
        public string PriceControlOption { get; set; }

        [CommandLineOption("preorder-control", DefaultValue = "", Description = "PreOrder control option")]
        public string PreOrderControlOption { get; set; }

        [CommandLineOption('t', "token", DefaultValue = "", Description = "CLI Token")]
        public override string TokenOption { get; set; }

        [CommandLineOption('u', "user", DefaultValue = "", Description = "Login ID of CLI Token")]
        public override string UserOption { get; set; }

        [CommandLineOption('p', "password", DefaultValue = "", Description = "Password of CLI Token")]
        public override string PasswordOption { get; set; }

        // 現時限の終了日時、次時限の開始日時
        private DateTime CurrentTimePeriodBegin;
        private DateTime CurrentTimePeriodEnd;
        private DateTime NextTimePeriodBegin;

        // 操作内容
        private enum Operation
        {
            None,       // 指定なし
            Begin,      // 開始する
            End,        // 終了する
        }

        // 価格情報の操作
        private Operation PriceOperation;
        private DateTime PriceTimePeriodBegin;

        // 予約販売の操作
        private Operation PreOrderOperation;
        private DateTime ReleaseTimePeriodBegin;

        protected bool SetupSaleOperations()
        {
            // 現時限の終了日時、次時限の開始日時
            var now = Setting.Current.StartDateTime;
            CurrentTimePeriodBegin = ToolUtility.ToTimePeriodBegin(now);
            CurrentTimePeriodEnd = ToolUtility.ToTimePeriodEnd(now);
            NextTimePeriodBegin = ToolUtility.ToTimePeriodNext(now);

            // 価格の開始日時
            if (!SetupOperation(out PriceOperation, out PriceTimePeriodBegin, NextTimePeriodBegin, PriceControlOption, "price-control"))
            {
                return false;
            }

            // 販売の開始日時
            if (!SetupOperation(out PreOrderOperation, out ReleaseTimePeriodBegin, PriceTimePeriodBegin, PreOrderControlOption, "preorder-control"))
            {
                return false;
            }
            return true;
        }

        private bool SetupOperation(out Operation operation, out DateTime begin, DateTime lower, string option, string name)
        {
            begin = lower;
            if (string.IsNullOrEmpty(option))
            {
                // 指定なし
                operation = Operation.None;
            }
            else if (string.Compare(option, "end", true) == 0)
            {
                // 終了指定
                operation = Operation.End;
            }
            else if (string.Compare(option, "auto", true) == 0)
            {
                // 自動設定
                operation = Operation.Begin;
            }
            else
            {
                // 日時指定
                operation = Operation.Begin;

                var value = ToolUtility.ToDateTime(option);
                begin = ToolUtility.ToTimePeriodBegin(value);
                if (value < lower)
                {
                    var date = lower.ToString("yyyy/MM/dd HH:mm");
                    Log.WriteLine($"Error: Invalid datetime. Please input '{name}' value after \"{date}\".");
                    return false;
                }
            }
            return begin >= lower;
        }

        // 価格登録処理 ->
        protected bool RegisterPriceByNsUid(string apiType, NsUidMap nsUids)
        {
            bool result = true;
            foreach (var nsUid in nsUids.ToNsUids())
            {
                result &= RegisterPriceByNsUid(apiType, nsUid, false, true);
            }
            return result;
        }

        protected bool RegisterPriceByNsUid(string apiType, string nsUid, bool isPreOrderable, bool isUpdatable)
        {
            var shop = new ShopHelper(Setting.Current);

            // 価格情報取得
            PriceInfo info = null;
            if (!shop.GetPrice(out info, apiType, nsUid))
            {
                return false;
            }

            // 価格情報登録
            if (!info.IsExistChangeSetId())
            {
                // 新規登録
                info = CreatePriceInfo(nsUid, isPreOrderable);
            }
            else
            {
                // 更新登録
                UpdatePriceInfo(info, isPreOrderable, isUpdatable);
            }
            return shop.RegisterPrice(apiType, info);
        }
        // <-

        private PriceInfo CreatePriceInfo(string nsUid, bool isPreOrderable)
        {
            var info = new PriceInfo();
            info.onlinePrices = new List<PriceInfo.OnlinePrice>();
            {
                var onlinePrice = new PriceInfo.OnlinePrice();
                onlinePrice.dataSource = "QA";
                onlinePrice.changeSetId = null;
                onlinePrice.approveStatus = null;
                onlinePrice.approveDate = null;
                onlinePrice.nsUid = nsUid;
                onlinePrice.prices = CreatePrices(isPreOrderable);

                info.onlinePrices.Add(onlinePrice);
            }
            return info;
        }

        private List<PriceInfo.OnlinePrice.Price> CreatePrices(bool isPreOrderable)
        {
            var prices = new List<PriceInfo.OnlinePrice.Price>();
            foreach (var country in Setting.Current.TargetCountries)
            {
                prices.Add(CreateNewPrice(country, isPreOrderable));
            }
            return prices;
        }

        private PriceInfo.OnlinePrice.Price CreateNewPrice(string country, bool isPreOrderable)
        {
            // 既定値の設定
            var price = new PriceInfo.OnlinePrice.Price();
            price.priceId = null;
            price.endDatetime = null;
            price.countryCode = country;
            price.amount = "0";
            price.registeredBmsFlag = "0";
            price.preOrderFlag = "0";
            if (country == "US")
            {
                price.startDatetime = "2016-12-08T07:00:00+0000";
                price.taxCategory = new PriceInfo.OnlinePrice.Price.TaxCategory();
                price.taxCategory.categoryId = "1001";
                price.taxCategory.transactionTypeId = "1";
            }
            else
            {
                price.startDatetime = "2017-01-13T02:00:00+0000";
                price.taxCategory = null;
            }

            // 価格情報の操作
            switch (PriceOperation)
            {
            case Operation.Begin:
                price.startDatetime = ToolUtility.ToIso8601Utc(PriceTimePeriodBegin);
                break;

            case Operation.End:
                price.endDatetime = ToolUtility.ToIso8601Utc(CurrentTimePeriodEnd);
                break;
            }

            // 予約販売の操作
            if (isPreOrderable)
            {
                switch (PreOrderOperation)
                {
                case Operation.Begin:
                    price.preOrderFlag = "1";
                    price.appReleaseControls = new PriceInfo.OnlinePrice.Price.AppReleaseControls();
                    price.appReleaseControls.autoSetFlag = "0";
                    price.appReleaseControls.releaseDatetime = ToolUtility.ToIso8601Utc(ReleaseTimePeriodBegin);
                    break;
                }
            }
            return price;
        }

        private void UpdatePriceInfo(PriceInfo info, bool isPreOrderable, bool isUpdatable)
        {
            foreach (var onlinePrice in info.onlinePrices)
            {
                // 更新時に下記は必ずクリアする
                onlinePrice.approveStatus = null;
                onlinePrice.approveDate = null;

                // 価格情報の操作なら更新必要
                var needUpdate = (PriceOperation != Operation.None);
                if (isPreOrderable)
                {
                    // 予約販売の操作なら更新必要
                    needUpdate |= (PreOrderOperation != Operation.None);
                }

                if (onlinePrice.prices == null || onlinePrice.prices.Count == 0)
                {
                    // 空なら既定値を新規登録
                    onlinePrice.prices = CreatePrices(isPreOrderable);
                }
                else if (isUpdatable && needUpdate)
                {
                    for (var i = 0; i < onlinePrice.prices.Count; i++)
                    {
                        var price = onlinePrice.prices[i];

                        // 対象国が指定されていて対象外ならスキップ
                        if (Setting.Current.HasTargetCountries &&
                            !Setting.Current.TargetCountries.Contains(price.countryCode))
                        {
                            continue;
                        }

                        var priceStart = DateTime.Parse(price.startDatetime);
                        if (priceStart < CurrentTimePeriodEnd)
                        {
                            // ※価格の開始日時が過ぎている場合は終了日時のみ変更可能

                            // 価格情報の操作なら既存価格を終了する
                            var end = (PriceOperation != Operation.None);
                            if (isPreOrderable)
                            {
                                // 予約販売開始なら既存価格を終了する
                                end |= (PreOrderOperation == Operation.Begin);

                                // 予約販売中で予約販売終了なら既存価格を終了する
                                end |= (PreOrderOperation == Operation.End && string.Compare(price.preOrderFlag, "1") == 0);
                            }

                            // 価格情報の開始なら新規価格を挿入する
                            var begin = (PriceOperation == Operation.Begin);
                            if (isPreOrderable)
                            {
                                // 予約販売開始なら新規価格を挿入する
                                begin |= (PreOrderOperation == Operation.Begin);
                            }

                            var insert = false;
                            if (end)
                            {
                                if (string.IsNullOrEmpty(price.endDatetime))
                                {
                                    // 未終了なら終了日時を設定
                                    price.endDatetime = ToolUtility.ToIso8601Utc(CurrentTimePeriodEnd);
                                    insert = begin;
                                }
                                else
                                {
                                    // 終了済みなら終了日時により分岐
                                    var priceEnd = DateTime.Parse(price.endDatetime);
                                    if (priceEnd > CurrentTimePeriodEnd)
                                    {
                                        // 終了日時が未来なら現時限を終了日時に変更
                                        price.endDatetime = ToolUtility.ToIso8601Utc(CurrentTimePeriodEnd);
                                        insert = begin;
                                    }
                                    else
                                    {
                                        // 終了日時が過去で次も同じ国なら次レコードで判定する
                                        var j = i + 1;
                                        if ((j == onlinePrice.prices.Count) ||
                                            (price.countryCode != onlinePrice.prices[j].countryCode))
                                        {
                                            insert = begin;
                                        }
                                    }
                                }
                            }
                            if (insert)
                            {
                                // 同じ国の既存価格を参考に進化価格を挿入
                                onlinePrice.prices.Insert(++i, CreateNextPrice(price, isPreOrderable));
                            }
                        }
                        else
                        {
                            // ※価格の開始日時が過ぎていない場合は全て変更可能
                            UpdatePrice(price, isPreOrderable);
                        }
                    }
                }
            }
        }

        private PriceInfo.OnlinePrice.Price CreateNextPrice(PriceInfo.OnlinePrice.Price prev, bool isPreOrderable)
        {
            var price = new PriceInfo.OnlinePrice.Price();
            price.priceId = null;
            price.startDatetime = ToolUtility.ToIso8601Utc(PriceTimePeriodBegin);
            price.endDatetime = null;
            price.countryCode = prev.countryCode;
            price.amount = "0";
            price.registeredBmsFlag = "0";
            price.preOrderFlag = "0";
            price.deferredFlag = null;
            price.taxCategory = null;
            price.appReleaseControls = null;
            price.privilegeCodes = null;
            price.privilegeGrants = null;

            // 下記のみ元の内容から複製
            if (prev.taxCategory != null)
            {
                price.taxCategory = new PriceInfo.OnlinePrice.Price.TaxCategory();
                price.taxCategory.categoryId = prev.taxCategory.categoryId;
                price.taxCategory.transactionTypeId = prev.taxCategory.transactionTypeId;
            }

            // 予約販売の操作
            if (isPreOrderable)
            {
                switch (PreOrderOperation)
                {
                case Operation.Begin:
                    price.preOrderFlag = "1";
                    price.appReleaseControls = new PriceInfo.OnlinePrice.Price.AppReleaseControls();
                    price.appReleaseControls.autoSetFlag = "0";
                    price.appReleaseControls.releaseDatetime = ToolUtility.ToIso8601Utc(ReleaseTimePeriodBegin);
                    break;
                }
            }
            return price;
        }

        private void UpdatePrice(PriceInfo.OnlinePrice.Price price, bool isPreOrderable)
        {
            // 価格情報の操作
            switch (PriceOperation)
            {
            case Operation.Begin:
                price.startDatetime = ToolUtility.ToIso8601Utc(PriceTimePeriodBegin);
                break;

            case Operation.End:
                price.startDatetime = ToolUtility.ToIso8601Utc(CurrentTimePeriodBegin);
                price.endDatetime = ToolUtility.ToIso8601Utc(CurrentTimePeriodEnd);
                break;
            }

            // 予約販売の操作
            if (isPreOrderable)
            {
                switch (PreOrderOperation)
                {
                case Operation.Begin:
                    var release = ReleaseTimePeriodBegin;
                    var start = DateTime.Parse(price.startDatetime);
                    if (release < start)
                    {
                        var src = release.ToString("yyyy/MM/dd HH:mm");
                        var dst = start.ToString("yyyy/MM/dd HH:mm");
                        Log.WriteLine($"Warning: Replaced preorder-control value of {price.countryCode}. {src} -> {dst}.");

                        // 発売日が開始日より早くなるので、開始日に置き換え
                        release = start;
                    }
                    price.preOrderFlag = "1";
                    price.appReleaseControls = new PriceInfo.OnlinePrice.Price.AppReleaseControls();
                    price.appReleaseControls.autoSetFlag = "0";
                    price.appReleaseControls.releaseDatetime = ToolUtility.ToIso8601Utc(release);
                    break;

                case Operation.End:
                    price.preOrderFlag = "0";
                    price.appReleaseControls = null;
                    break;
                }
            }
        }
    }
}
