﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using Nintendo.Foundation.IO;
using ContentsUploader.Assistants;

namespace ContentsUploader.Commands
{
    public abstract class CommandBase
    {
        // 実行モード
        protected enum Mode
        {
            None = 0,
            IsTokenRequired = 1 << 0,
        }

        // サービス環境指定
        [CommandLineOption('e', "environment", Description = "The environment used for service. Now you can only use \"td1\" and \"jd1\".", IsRequired = true)]
        public string EnvironmentAlias { get; set; }

        // プロキシ指定
        [CommandLineOption("proxy", DefaultValue = "", Description = "The proxy to be used to access the Internet. e.g.http://proxy:8080.")]
        public string ProxyOption { get; set; }

        // 詳細ログ表示指定
        [CommandLineOption('v', "verbose", DefaultValue = false, Description = "Show detail log.")]
        public bool IsVerbose { get; set; }

        // 旧環境指定（※特定の環境を指すものではなく、td1 jd1 間で相違が発生した場合に利用します）
        [CommandLineOption("old-env", DefaultValue = false, IsHidden = true, Description = "Old Environment")]
        public bool IsOldEnvironment { get; set; }

        // プロパティ
        public virtual string NspPathOption { get; set; }
        public virtual string ContentMetaIdOption { get; set; }
        public virtual string ApplicationIdOption { get; set; }
        public virtual string RomIdOption { get; set; }

        public virtual string TokenOption { get; set; }
        public virtual string UserOption { get; set; }
        public virtual string PasswordOption { get; set; }

        // ディレクトリ指定の有無
        protected virtual bool HasNspDirectoryOption()
        {
            var path = NspPathOption;
            return !string.IsNullOrEmpty(path) && Directory.Exists(path);
        }

        // ファイル指定の有無
        protected virtual bool HasNspFileOption()
        {
            var path = NspPathOption;
            return !string.IsNullOrEmpty(path) && !Directory.Exists(path);
        }

        // ContentMetaId 指定の有無
        protected virtual bool HasContentMetaIdOption()
        {
            var id = ContentMetaIdOption;
            return !string.IsNullOrEmpty(id);
        }

        // ApplicationId 指定の有無
        protected virtual bool HasApplicationIdOption()
        {
            var id = ApplicationIdOption;
            return !string.IsNullOrEmpty(id);
        }

        // RomId 指定の有無
        protected virtual bool HasRomIdOption()
        {
            var id = RomIdOption;
            return !string.IsNullOrEmpty(id);
        }

        // 上記以外のコマンド固有指定の有無
        protected virtual bool HasOtherOption()
        {
            return false;
        }

        // コマンド実行
        public abstract void Run();

        protected virtual void Run(string name, Mode mode = Mode.None)
        {
            NspPathOption = !string.IsNullOrEmpty(NspPathOption) ? NspPathOption.Trim() : string.Empty;
            using (var command = new CommandStatus(name, IsVerbose))
            {
                var created = Setting.CreateCurrent(EnvironmentAlias, ProxyOption, IsVerbose);
                if (!(ValidateOptions() && created))
                {
                    return;
                }

                if (mode.HasFlag(Mode.IsTokenRequired) &&
                    !Setting.Current.PublishToken(TokenOption, UserOption, PasswordOption))
                {
                    return;
                }

                if (HasNspDirectoryOption())
                {
                    command.Result = RunByDirectory();
                }
                else if (HasNspFileOption())
                {
                    command.Result = RunByFile();
                }
                else if (HasContentMetaIdOption())
                {
                    command.Result = RunByContentMetaId();
                }
                else if (HasApplicationIdOption())
                {
                    command.Result = RunByApplicationId();
                }
                else if (HasRomIdOption())
                {
                    command.Result = RunByRomId();
                }
                else if (HasOtherOption())
                {
                    command.Result = RunByOther();
                }
                else
                {
                    Log.WriteLine($"Error: Please input the setting");
                }
            }
        }

        protected virtual bool ValidateOptions()
        {
            return true;
        }

        protected virtual bool RunByDirectory()
        {
            var path = string.Empty;
            if (!ToolUtility.ConvertToAbsoluteDirectoryPath(out path, NspPathOption))
            {
                Log.WriteLine($"Error: Directory not found. Please check the path \"{path}\".");
                return false;
            }

            var nsps = NspAccessor.ReadDirectory(path, IsVerbose);
            if (nsps.Count == 0)
            {
                Log.WriteLine($"Error: Nsp file not found. Please check the path \"{path}\".");
                return false;
            }

            bool result = true;
            foreach (var nsp in nsps)
            {
                result &= RunByFile(nsp);
            }
            return result;
        }

        protected virtual bool RunByFile()
        {
            var path = string.Empty;
            if (!ToolUtility.ConvertToAbsoluteNspFilePath(out path, NspPathOption))
            {
                Log.WriteLine($"Error: Nsp file not found. Please check the path \"{path}\".");
                return false;
            }

            var nsp = new NspAccessor(path, IsVerbose);
            return RunByFile(nsp);
        }

        protected virtual bool RunByFile(NspAccessor nsp)
        {
            bool result = true;
            foreach (var content in nsp.NspInnerContent)
            {
                result &= RunByContentMetaId(content.ContentMetaId);
            }
            return result;
        }

        protected virtual bool RunByContentMetaId()
        {
            var contentMetaId = new Id64(ContentMetaIdOption);
            return RunByContentMetaId(contentMetaId);
        }

        protected virtual bool RunByContentMetaId(Id64 contentMetaId)
        {
            throw new Exception("Unimplemented.");
        }

        protected virtual bool RunByApplicationId()
        {
            var applicationId = new Id64(ApplicationIdOption);
            return RunByApplicationId(applicationId);
        }

        protected virtual bool RunByApplicationId(Id64 applicationId)
        {
            throw new Exception("Unimplemented.");
        }

        protected virtual bool RunByRomId()
        {
            return RunByRomId(RomIdOption);
        }

        protected virtual bool RunByRomId(string romId)
        {
            throw new Exception("Unimplemented.");
        }

        protected virtual bool RunByOther()
        {
            throw new Exception("Unimplemented.");
        }
    }
}
